package template

import (
	"errors"
	"fmt"
	"io"
	"path"
	"strings"

	"a.yandex-team.ru/infra/hostctl/internal/unit"
	"a.yandex-team.ru/infra/hostctl/internal/unit/kind"
	"a.yandex-team.ru/infra/hostctl/pkg/pbutil"
	"a.yandex-team.ru/infra/hostctl/pkg/unitstorage"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"google.golang.org/protobuf/proto"
)

// BUG(warwish): FromReader creates reader not suitable for handling fragmented units.

func unitAbsent(meta *pb.ObjectMeta) bool {
	return meta.DeleteRequested || meta.Annotations["stage"] == "absent"
}

// template encapsulates rendering pipeline details needed for unit-file -> Unit transformation
type template struct {
	doc   unitstorage.Document
	name  string
	cCtx  CompiledCtx
	tMeta MetaTemplate
	s     unitstorage.Storage
}

// FromStorage creates template from unit name and underlying storage
func FromStorage(s unitstorage.Storage, name string) (Template, error) {
	doc, err := DocumentFromStorage(s, fmt.Sprintf("%s.yaml", name))
	if err != nil {
		return nil, err
	}
	ctxDef, err := ParseCtxDocument(doc)
	if err != nil {
		return nil, err
	}
	c := NewSimpleCtxCompiler()
	cCtx, err := c.Compile(ctxDef)
	if err != nil {
		return nil, err
	}
	tMeta, err := ParseMetaDocument(doc)
	if err != nil {
		return nil, err
	}
	return &template{
		doc:   doc,
		name:  tMeta.(*MetaPbTemplate).Name,
		cCtx:  cCtx,
		tMeta: tMeta,
		s:     s,
	}, nil
}

// FromReader creates template from unit name and underlying storage
func FromReader(r io.Reader, name string, repo string) (Template, error) {
	doc, err := DocumentFromReader(r, fmt.Sprintf("%s.yaml", name), repo)
	if err != nil {
		return nil, err
	}
	ctxDef, err := ParseCtxDocument(doc)
	if err != nil {
		return nil, err
	}
	c := NewSimpleCtxCompiler()
	cCtx, err := c.Compile(ctxDef)
	if err != nil {
		return nil, err
	}
	tMeta, err := ParseMetaDocument(doc)
	if err != nil {
		return nil, err
	}
	return &template{
		doc:   doc,
		name:  tMeta.(*MetaPbTemplate).Name,
		cCtx:  cCtx,
		tMeta: tMeta,
	}, nil
}

// Render renders template for concrete HostInfo
func (t *template) Render(hi *pb.HostInfo) (*unit.Unit, error) {
	mCtx, err := t.cCtx.Materialize(hi)
	if err != nil {
		return nil, err
	}
	mMeta, err := t.tMeta.Materialize(mCtx)
	if err != nil {
		return nil, err
	}
	sp := NewDefaultSpecParser()
	tSpec, err := sp.ParseSpec(t.doc, mMeta)
	if err != nil {
		return nil, err
	}
	mSpec, err := tSpec.Materialize(mCtx)
	if err != nil {
		return nil, err
	}
	err = augmentFragments(t, mMeta, mSpec, hi)
	if err != nil {
		return nil, err
	}
	id, err := pbutil.PbDigest(mSpec)
	if err != nil {
		return nil, fmt.Errorf("failed to create id for '%s': %w", mMeta.Name, err)
	}
	s, err := parseSkipRemovePhase((*pb.ObjectMeta)(mMeta))
	if err != nil {
		return nil, err
	}
	return unit.RenderedUnit(
		mSpec,
		id, mMeta.Name, kind.Kind(mMeta.Kind),
		&pb.SlotMeta{
			Labels:          mMeta.Labels,
			Annotations:     mMeta.Annotations,
			XKind:           mMeta.Kind,
			SkipRemovePhase: s,
		}, &pb.RevisionMeta{
			Kind:    mMeta.Kind,
			Version: mMeta.Version,
		}, unitAbsent((*pb.ObjectMeta)(mMeta))), nil
}

func (t *template) Name() string {
	return t.name
}

func (t *template) Path() string {
	return t.doc.Path()
}

type removedTemplate struct {
	name string
}

func (t *removedTemplate) Path() string {
	return "<render>"
}

func (t *removedTemplate) Name() string {
	return t.name
}

// augmentFragments renders and merges fragments of unit to provided mSpec
func augmentFragments(t *template, mMeta *MaterializedMeta, mSpec proto.Message, hi *pb.HostInfo) error {
	unitFragments, err := renderFragments(t, mMeta, hi)
	if err != nil {
		return err
	}
	for _, f := range unitFragments {
		pbPod := mSpec.(*pb.HostPodSpec)
		mergeFragment(pbPod, mMeta, f)
	}
	return nil
}

// renderFragments loads and renders fragments from mMeta annotations using provided pb.HostInfo and template
func renderFragments(t *template, mMeta *MaterializedMeta, hi *pb.HostInfo) ([]*unit.Unit, error) {
	var unitFragments []*unit.Unit
	fStr, ok := mMeta.Annotations["fragments"]
	if !ok || len(fStr) == 0 {
		return nil, nil
	}
	if t.s == nil {
		return nil, errors.New("failed to render fragments of unit %s, storage is nil")
	}

	fragments := strings.Fields(fStr)
	for _, f := range fragments {
		t, err := FromStorage(t.s, path.Join(fmt.Sprintf("%s.d", mMeta.Name), f))
		if err != nil {
			return nil, fmt.Errorf("failed to load fragment %s: %w", f, err)
		}
		u, err := t.Render(hi)
		if err != nil {
			return nil, fmt.Errorf("failed to render fragment %s: %w", f, err)
		}
		unitFragments = append(unitFragments, u)
	}
	return unitFragments, nil
}

// mergeFragment appends entities from fragment unit to pod spec and infers pod meta with fragment stage
func mergeFragment(pod *pb.HostPodSpec, podMeta *MaterializedMeta, frag *unit.Unit) {
	fStage := fmt.Sprintf("fragments.stage/%s", frag.Name())
	podMeta.Annotations[fStage] = frag.SlotMeta().Annotations["stage"]
	if !frag.Absent() {
		f := frag.Spec().(*pb.HostPodSpec)
		pod.Files = append(pod.Files, f.Files...)
		pod.Packages = append(pod.Packages, f.Packages...)
		pod.PortoDaemons = append(pod.PortoDaemons, f.PortoDaemons...)
		pod.Services = append(pod.Services, f.Services...)
		pod.Timers = append(pod.Timers, f.Timers...)
	}
}

// Removed creates a special template which applies to slot during slot removal
func Removed(name string) Template {
	return &removedTemplate{name: name}
}

func (t *removedTemplate) Render(_ *pb.HostInfo) (*unit.Unit, error) {
	m := &pb.ObjectMeta{
		Kind:        "PackageSet", // do not care
		Name:        t.name,
		Annotations: map[string]string{"stage": "absent"},
	}
	s := &pb.PackageSetSpec{}
	return unit.RenderedUnit(
		s,
		"<removed>", m.Name, kind.Kind(m.Kind),
		&pb.SlotMeta{
			Annotations:     m.Annotations,
			XKind:           m.Kind,
			SkipRemovePhase: &pb.SkipRemovePhase{},
		}, &pb.RevisionMeta{
			Kind:    m.Kind,
			Version: m.Version,
		}, true), nil
}

type testTemplate struct {
	meta *pb.ObjectMeta
	spec proto.Message
}

func (t *testTemplate) Render(_ *pb.HostInfo) (*unit.Unit, error) {
	id, err := pbutil.PbDigest(t.spec)
	if err != nil {
		return nil, fmt.Errorf("failed to create id for '%s': %w", t.meta.Name, err)
	}
	return unit.RenderedUnit(
		t.spec,
		id, t.meta.Name, kind.Kind(t.meta.Kind),
		&pb.SlotMeta{
			Annotations:     t.meta.Annotations,
			XKind:           t.meta.Kind,
			SkipRemovePhase: &pb.SkipRemovePhase{},
		}, &pb.RevisionMeta{
			Kind:    t.meta.Kind,
			Version: t.meta.Version,
		}, unitAbsent(t.meta)), nil
}

func (t *testTemplate) Path() string {
	return "<test>"
}

func (t *testTemplate) Name() string {
	return t.meta.Name
}

func NewTestTemplate(meta *pb.ObjectMeta, spec proto.Message) Template {
	return &testTemplate{meta: meta, spec: spec}
}
