package template

import (
	"reflect"
	"strings"

	"a.yandex-team.ru/infra/hostctl/internal/pyfmt"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"google.golang.org/protobuf/proto"
)

// interpolatePb recursively substitutes strings in given proto.Message with variables from ctxMap
func interpolatePb(ctxMap map[string]string, m proto.Message) error {
	return interpolateStruct(ctxMap, reflect.ValueOf(m).Elem())
}

// interpolateStruct recursively substitutes strings in given struct with variables from ctxMap
func interpolateStruct(ctxMap map[string]string, val reflect.Value) error {
	// skip zero Value's fields
	if !val.IsValid() {
		return nil
	}
	for i := 0; i < val.NumField(); i++ {
		ft := val.Type().Field(i)
		if ft.Anonymous || ft.PkgPath != "" {
			continue
		}
		switch f := val.Field(i); f.Kind() {
		case reflect.Slice:
			err := interpolateSlice(ctxMap, f)
			if err != nil {
				return err
			}
		case reflect.String:
			err := interpolateString(ctxMap, f)
			if err != nil {
				return err
			}
		case reflect.Struct:
			err := interpolateStruct(ctxMap, f)
			if err != nil {
				return err
			}
		case reflect.Ptr:
			err := interpolateStruct(ctxMap, f.Elem())
			if err != nil {
				return err
			}
		}
	}
	return nil
}

// interpolateSlice recursively substitutes strings in given slice with variables from ctxMap
func interpolateSlice(ctxMap map[string]string, val reflect.Value) error {
	for j := 0; j < val.Len(); j++ {
		switch v := val.Index(j); v.Kind() {
		case reflect.Slice:
			err := interpolateSlice(ctxMap, v)
			if err != nil {
				return err
			}
		case reflect.Struct:
			err := interpolateStruct(ctxMap, v)
			if err != nil {
				return err
			}
		case reflect.String:
			err := interpolateString(ctxMap, v)
			if err != nil {
				return err
			}
		case reflect.Ptr:
			err := interpolateStruct(ctxMap, v.Elem())
			if err != nil {
				return err
			}
		}
	}
	return nil
}

// interpolateString substitutes reflected string value (must be exported field of struct)
// with variables from ctxMap
func interpolateString(ctxMap map[string]string, val reflect.Value) error {
	v, err := fmtString(val.String(), ctxMap)
	if err != nil {
		return err
	}
	val.Set(reflect.ValueOf(v))
	return nil
}

// fmtString substitutes given string with ctxMap variables
func fmtString(s string, ctxMap map[string]string) (string, error) {
	if !strings.Contains(s, "{") {
		return s, nil
	}
	return pyfmt.Fmt(s, ctxMap)
}

// fmtMap substitutes given map values with variables from ctxMap
func fmtMap(m map[string]string, ctxMap map[string]string) error {
	for k, v := range m {
		s, err := fmtString(v, ctxMap)
		if err != nil {
			return err
		}
		m[k] = s
	}
	return nil
}

// normalizeSpec:
//   * remove skipped packages from specs
//   * remove empty params from PortoDaemon cmd lime
func normalizeSpec(s proto.Message) {
	switch spec := s.(type) {
	case *pb.PackageSetSpec:
		spec.Packages = filterSkippedPackages(spec.Packages)
	case *pb.PortoDaemon:
		spec.Packages = filterSkippedPackages(spec.Packages)
		spec.Properties.Cmd = removeEmpty(spec.Properties.Cmd)
	case *pb.SystemServiceSpec:
		spec.Packages = filterSkippedPackages(spec.Packages)
	case *pb.TimerJobSpec:
		spec.Packages = filterSkippedPackages(spec.Packages)
	case *pb.HostPodSpec:
		spec.Packages = filterSkippedPackages(spec.Packages)
		for _, pDaemon := range spec.PortoDaemons {
			pDaemon.Properties.Cmd = removeEmpty(pDaemon.Properties.Cmd)
		}

	}
}

// removeEmpty removes empty strings from slice preserving order
func removeEmpty(arr []string) []string {
	res := make([]string, 0, len(arr))
	for _, s := range arr {
		if s == "" {
			continue
		}
		res = append(res, s)
	}
	return res
}

const PkgVersionSkip = "<skip>"

// filterSkippedPackages
// Inplace filter packages with "<skip>" version. Without alloc tmp slice.
func filterSkippedPackages(pkgs []*pb.SystemPackage) []*pb.SystemPackage {
	j := 0
	for i := 0; i < len(pkgs); i++ {
		pkg := pkgs[i]
		if pkg.Version != PkgVersionSkip {
			pkgs[j] = pkg
			j++
		}
	}
	return pkgs[:j]
}
