package template

import (
	"reflect"
	"testing"

	pb "a.yandex-team.ru/infra/hostctl/proto"
	"github.com/stretchr/testify/assert"
)

func TestFilterSkippedPackages(t *testing.T) {
	pkgs := []*pb.SystemPackage{
		{
			Name:    "mock1",
			Version: "<skip>",
		},
		{
			Name:    "mock2",
			Version: "not-skip",
		},
	}
	expected := []*pb.SystemPackage{
		{
			Name:    "mock2",
			Version: "not-skip",
		},
	}
	assert.Equal(t, expected, filterSkippedPackages(pkgs))
}

func TestRemoveEmpty(t *testing.T) {
	a := []string{"mock", "", "mock2"}
	assert.Equal(t, []string{"mock", "mock2"}, removeEmpty(a))
}

func TestNormalizeSpec_PackageSet(t *testing.T) {
	s := &pb.PackageSetSpec{
		Packages: []*pb.SystemPackage{
			{
				Name:    "mock",
				Version: "<skip>",
			},
			{
				Name:    "mock2",
				Version: "mock",
			},
		},
	}
	expected := &pb.PackageSetSpec{
		Packages: []*pb.SystemPackage{
			{
				Name:    "mock2",
				Version: "mock",
			},
		},
	}
	normalizeSpec(s)
	assert.Equal(t, expected, s)
}

func TestNormalizeSpec_PortoDaemon(t *testing.T) {
	s := &pb.PortoDaemon{
		Packages: []*pb.SystemPackage{
			{
				Name:    "mock",
				Version: "<skip>",
			},
			{
				Name:    "mock2",
				Version: "mock",
			},
		},
		Properties: &pb.PortoProperties{
			Cmd: []string{"/bin/rm", "-Rf", "/*", ""},
		},
	}
	expected := &pb.PortoDaemon{
		Packages: []*pb.SystemPackage{
			{
				Name:    "mock2",
				Version: "mock",
			},
		},
		Properties: &pb.PortoProperties{
			Cmd: []string{"/bin/rm", "-Rf", "/*"},
		},
	}
	normalizeSpec(s)
	assert.Equal(t, expected, s)
}

func TestNormalizeSpec_SystemService(t *testing.T) {
	s := &pb.SystemServiceSpec{
		Packages: []*pb.SystemPackage{
			{
				Name:    "mock",
				Version: "<skip>",
			},
			{
				Name:    "mock2",
				Version: "mock",
			},
		},
	}
	expected := &pb.SystemServiceSpec{
		Packages: []*pb.SystemPackage{
			{
				Name:    "mock2",
				Version: "mock",
			},
		},
	}
	normalizeSpec(s)
	assert.Equal(t, expected, s)
}

func TestNormalizeSpec_TimerJob(t *testing.T) {
	s := &pb.TimerJobSpec{
		Packages: []*pb.SystemPackage{
			{
				Name:    "mock",
				Version: "<skip>",
			},
			{
				Name:    "mock2",
				Version: "mock",
			},
		},
	}
	expected := &pb.TimerJobSpec{
		Packages: []*pb.SystemPackage{
			{
				Name:    "mock2",
				Version: "mock",
			},
		},
	}
	normalizeSpec(s)
	assert.Equal(t, expected, s)
}

func TestNormalizeSpec_HostPod(t *testing.T) {
	s := &pb.HostPodSpec{
		Packages: []*pb.SystemPackage{
			{
				Name:    "mock",
				Version: "<skip>",
			},
			{
				Name:    "mock2",
				Version: "mock",
			},
		},
		PortoDaemons: []*pb.HostPodPortoDaemon{
			{
				Properties: &pb.PortoProperties{
					Cmd: []string{"/bin/rm", "-Rf", "/*", ""},
				},
			},
		},
	}
	expected := &pb.HostPodSpec{
		Packages: []*pb.SystemPackage{
			{
				Name:    "mock2",
				Version: "mock",
			},
		},
		PortoDaemons: []*pb.HostPodPortoDaemon{
			{
				Properties: &pb.PortoProperties{
					Cmd: []string{"/bin/rm", "-Rf", "/*"},
				},
			},
		},
	}
	normalizeSpec(s)
	assert.Equal(t, expected, s)
}

func TestFmtMap(t *testing.T) {
	m := map[string]string{
		"k1": "v1",
		"k2": "{var2}",
	}
	ctx := map[string]string{
		"var2": "mock2",
	}
	expected := map[string]string{
		"k1": "v1",
		"k2": "mock2",
	}
	err := fmtMap(m, ctx)
	assert.NoError(t, err)
	assert.Equal(t, expected, m)
	m["k1"] = "{malformed"
	err = fmtMap(m, ctx)
	assert.Error(t, err)
}

func TestFmtString(t *testing.T) {
	ctx := map[string]string{"var1": "val1"}
	s := "mock"
	r, err := fmtString(s, ctx)
	assert.NoError(t, err)
	assert.Equal(t, "mock", r)
	s = "mock-{var1}"
	r, err = fmtString(s, ctx)
	assert.NoError(t, err)
	assert.Equal(t, "mock-val1", r)
	s = "{malformed"
	r, err = fmtString(s, ctx)
	assert.Error(t, err)
	assert.Equal(t, "", r)
	s = "{var2}"
	r, err = fmtString(s, ctx)
	assert.Error(t, err)
	assert.Equal(t, "", r)
}

func TestInterpolateString(t *testing.T) {
	ctx := map[string]string{"var1": "val1"}
	s := &struct {
		F string
	}{"mock-{var1}"}
	r := reflect.ValueOf(s)
	err := interpolateString(ctx, r.Elem().Field(0))
	assert.NoError(t, err)
	assert.Equal(t, "mock-val1", s.F)
}

func TestInterpolateSlice(t *testing.T) {
	ctx := map[string]string{"var1": "val1", "var2": "val2"}
	type ts struct {
		F  string
		F2 string
	}
	s := []ts{{"{var1}", "f2"}, {"f1", "{var2}"}}
	r := reflect.ValueOf(s)
	err := interpolateSlice(ctx, r)
	assert.NoError(t, err)
	assert.Equal(t, []ts{{"val1", "f2"}, {"f1", "val2"}}, s)

	s2 := []*ts{{"{var1}", "f2"}, {"f1", "{var2}"}}
	r = reflect.ValueOf(s2)
	err = interpolateSlice(ctx, r)
	assert.NoError(t, err)
	assert.Equal(t, []*ts{{"val1", "f2"}, {"f1", "val2"}}, s2)

	s3 := []string{"{var1}", "mock"}
	r = reflect.ValueOf(s3)
	err = interpolateSlice(ctx, r)
	assert.NoError(t, err)
	assert.Equal(t, []string{"val1", "mock"}, s3)

	s4 := [][]string{{"{var1}", "mock"}, {"{var2}", "mock2"}}
	r = reflect.ValueOf(s4)
	err = interpolateSlice(ctx, r)
	assert.NoError(t, err)
	assert.Equal(t, [][]string{{"val1", "mock"}, {"val2", "mock2"}}, s4)
}

func TestInterpolateStruct(t *testing.T) {
	ctx := map[string]string{"var1": "val1", "var2": "val2"}
	type tsInt struct {
		F  string
		F2 string
	}
	type tsExt struct {
		F  string
		F2 string
		I  tsInt
		I2 *tsInt
		S  []tsInt
		S2 []*tsInt
		N  []tsInt
		N2 []*tsInt
	}
	s := &tsExt{
		F:  "{var1}",
		F2: "mock",
		I: tsInt{
			F:  "{var2}",
			F2: "mock-int2",
		},
		I2: &tsInt{
			F:  "{var1}",
			F2: "mock-int-p2",
		},
		S:  []tsInt{{"{var2}", "mock-int-s-2"}},
		S2: []*tsInt{{"mock-int-sp-1", "{var1}"}},
	}
	expected := &tsExt{
		F:  "val1",
		F2: "mock",
		I: tsInt{
			F:  "val2",
			F2: "mock-int2",
		},
		I2: &tsInt{
			F:  "val1",
			F2: "mock-int-p2",
		},
		S:  []tsInt{{"val2", "mock-int-s-2"}},
		S2: []*tsInt{{"mock-int-sp-1", "val1"}},
	}
	r := reflect.ValueOf(s)
	err := interpolateStruct(ctx, r.Elem())
	assert.NoError(t, err)
	assert.Equal(t, expected, s)
}

func TestInterpolatePb(t *testing.T) {
	ctx := map[string]string{"var1": "val1", "var2": "val2"}
	s := &pb.ManagedFile{Path: "/{var1}/{var2}"}
	err := interpolatePb(ctx, s)
	assert.NoError(t, err)
	assert.Equal(t, &pb.ManagedFile{Path: "/val1/val2"}, s)
}
