package template

import (
	"testing"

	"a.yandex-team.ru/infra/hostctl/internal/unit"
	"a.yandex-team.ru/infra/hostctl/internal/unit/kind"
	"a.yandex-team.ru/infra/hostctl/pkg/unitstorage"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/test/assertpb"
	"github.com/stretchr/testify/assert"
)

func TestMergeFragment(t *testing.T) {
	podMeta := &MaterializedMeta{
		Annotations: make(map[string]string),
	}
	pod := &pb.HostPodSpec{
		Packages:     []*pb.SystemPackage{{Name: "mock", Version: "vmock"}},
		Files:        []*pb.ManagedFile{{Path: "/mock", Content: "mock"}},
		PortoDaemons: []*pb.HostPodPortoDaemon{{Name: "blah", Properties: &pb.PortoProperties{Cmd: []string{"/bin/sleep"}}}},
		Services:     []*pb.HostPodService{{Name: "smock"}},
		Timers:       []*pb.HostPodTimer{{Name: "tmock"}},
	}
	fragMeta := &pb.SlotMeta{XKind: string(kind.HostPodFragment), Annotations: map[string]string{"stage": "test"}}
	frag := &pb.HostPodSpec{
		Packages:     []*pb.SystemPackage{{Name: "frag-mock", Version: "vmock"}},
		Files:        []*pb.ManagedFile{{Path: "/frag/mock", Content: "mock"}},
		PortoDaemons: []*pb.HostPodPortoDaemon{{Name: "frag-blah", Properties: &pb.PortoProperties{Cmd: []string{"/bin/sleep"}}}},
		Services:     []*pb.HostPodService{{Name: "frag-smock"}},
		Timers:       []*pb.HostPodTimer{{Name: "frag-tmock"}},
	}
	u := unit.NewTestUnit(frag, "fragment", "f1", kind.HostPodFragment, fragMeta, nil, false)
	mergeFragment(pod, podMeta, u)
	expected := &pb.HostPodSpec{
		Packages:     []*pb.SystemPackage{{Name: "mock", Version: "vmock"}, {Name: "frag-mock", Version: "vmock"}},
		Files:        []*pb.ManagedFile{{Path: "/mock", Content: "mock"}, {Path: "/frag/mock", Content: "mock"}},
		PortoDaemons: []*pb.HostPodPortoDaemon{{Name: "blah", Properties: &pb.PortoProperties{Cmd: []string{"/bin/sleep"}}}, {Name: "frag-blah", Properties: &pb.PortoProperties{Cmd: []string{"/bin/sleep"}}}},
		Services:     []*pb.HostPodService{{Name: "smock"}, {Name: "frag-smock"}},
		Timers:       []*pb.HostPodTimer{{Name: "tmock"}, {Name: "frag-tmock"}},
	}
	assertpb.Equal(t, expected, pod)
}

func TestMergeFragmentAbsent(t *testing.T) {
	podMeta := &MaterializedMeta{
		Annotations: make(map[string]string),
	}
	pod := &pb.HostPodSpec{
		Packages:     []*pb.SystemPackage{{Name: "mock", Version: "vmock"}},
		Files:        []*pb.ManagedFile{{Path: "/mock", Content: "mock"}},
		PortoDaemons: []*pb.HostPodPortoDaemon{{Name: "blah", Properties: &pb.PortoProperties{Cmd: []string{"/bin/sleep"}}}},
		Services:     []*pb.HostPodService{{Name: "smock"}},
		Timers:       []*pb.HostPodTimer{{Name: "tmock"}},
	}
	fragMeta := &pb.SlotMeta{XKind: string(kind.HostPodFragment), Annotations: map[string]string{"stage": "absent"}}
	frag := &pb.HostPodSpec{
		Packages:     []*pb.SystemPackage{{Name: "frag-mock", Version: "vmock"}},
		Files:        []*pb.ManagedFile{{Path: "/frag/mock", Content: "mock"}},
		PortoDaemons: []*pb.HostPodPortoDaemon{{Name: "frag-blah", Properties: &pb.PortoProperties{Cmd: []string{"/bin/sleep"}}}},
		Services:     []*pb.HostPodService{{Name: "frag-smock"}},
		Timers:       []*pb.HostPodTimer{{Name: "frag-tmock"}},
	}
	u := unit.NewTestUnit(frag, "fragment", "f1", kind.HostPodFragment, fragMeta, nil, true)
	mergeFragment(pod, podMeta, u)
	expected := &pb.HostPodSpec{
		Packages:     []*pb.SystemPackage{{Name: "mock", Version: "vmock"}},
		Files:        []*pb.ManagedFile{{Path: "/mock", Content: "mock"}},
		PortoDaemons: []*pb.HostPodPortoDaemon{{Name: "blah", Properties: &pb.PortoProperties{Cmd: []string{"/bin/sleep"}}}},
		Services:     []*pb.HostPodService{{Name: "smock"}},
		Timers:       []*pb.HostPodTimer{{Name: "tmock"}},
	}
	assertpb.Equal(t, expected, pod)
}

func TestFragmentedUnit(t *testing.T) {
	fs := unitstorage.NewMemFS([]*unitstorage.MemFile{
		{
			Path: "/etc/hostman/units.d/unit.yaml",
			Content: `meta:
  kind: HostPod
  version: "mock"
  name: unit
  annotations:
    fragments: "fragment"
spec:
  files:
    - path: /mock
      content: "mock"
  packages:
    - name: mock
      version: mock
  services:
    - name: mock
  timers:
    - name: mock
  porto_daemons:
    - name: mock
      properties:
        cmd:
          - "/bin/false"
`,
		},
		{
			Path: "/etc/hostman/units.d/unit.d/fragment.yaml",
			Content: `meta:
  kind: HostPodFragment
  version: "mock"
  name: fragment
  annotations:
    stage: "default"
spec:
  files:
    - path: /frag/mock
      content: "mock"
  packages:
    - name: frag-mock
      version: mock
  services:
    - name: frag-mock
  timers:
    - name: frag-mock
  porto_daemons:
    - name: frag-mock
      properties:
        cmd:
          - "/bin/true"
`,
		},
	})
	s, err := unitstorage.NewDefaultFSStorage(fs, "/", nil)
	assert.NoError(t, err)
	template, err := FromStorage(s, "unit")
	assert.NoError(t, err)
	hi := &pb.HostInfo{KernelRelease: "2.6.32-100500", Hostname: "mock.search.yandex.net"}
	u, err := template.Render(hi)
	assert.NoError(t, err)
	assert.Equal(t, kind.HostPod, u.Kind())
	expected := &pb.HostPodSpec{
		Packages:     []*pb.SystemPackage{{Name: "mock", Version: "mock"}, {Name: "frag-mock", Version: "mock"}},
		Files:        []*pb.ManagedFile{{Path: "/mock", Content: "mock"}, {Path: "/frag/mock", Content: "mock"}},
		PortoDaemons: []*pb.HostPodPortoDaemon{{Name: "mock", Properties: &pb.PortoProperties{Cmd: []string{"/bin/false"}}}, {Name: "frag-mock", Properties: &pb.PortoProperties{Cmd: []string{"/bin/true"}}}},
		Services:     []*pb.HostPodService{{Name: "mock"}, {Name: "frag-mock"}},
		Timers:       []*pb.HostPodTimer{{Name: "mock"}, {Name: "frag-mock"}},
	}
	assertpb.Equal(t, expected, u.Spec())
}
