package unit

import (
	"fmt"
	"strings"

	"a.yandex-team.ru/infra/hostctl/internal/unit/kind"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/infra/hostctl/internal/fmtutil"
	"a.yandex-team.ru/infra/hostctl/internal/orly"
	"a.yandex-team.ru/infra/hostctl/internal/unit/valid"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/yamlutil"
	"a.yandex-team.ru/infra/hostctl/pkg/pbutil"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/core/log"
)

func NewTestUnit(spec proto.Message, digest, name string, kind kind.Kind, slotMeta *pb.SlotMeta, revMeta *pb.RevisionMeta, absent bool) *Unit {
	return &Unit{
		spec:     spec,
		digest:   digest,
		name:     name,
		kind:     kind,
		slotMeta: slotMeta,
		revMeta:  revMeta,
		absent:   absent,
	}
}

/*
Unit represents managed unit described in .yaml or protobuf and read from FS/remote.
It can be viewed as incoming revision of a unit with metadata attached to it.
*/
type Unit struct {
	spec     proto.Message
	digest   string
	name     string
	kind     kind.Kind
	slotMeta *pb.SlotMeta
	revMeta  *pb.RevisionMeta
	absent   bool
}

func RenderedUnit(spec proto.Message, digest string, name string, kind kind.Kind, slotMeta *pb.SlotMeta, revMeta *pb.RevisionMeta, absent bool) *Unit {
	return &Unit{
		spec:     spec,
		digest:   digest,
		name:     name,
		kind:     kind,
		slotMeta: slotMeta,
		revMeta:  revMeta,
		absent:   absent,
	}
}

func (u *Unit) Spec() proto.Message {
	return u.spec
}

func (u *Unit) ID() string {
	return u.digest
}

func (u *Unit) Name() string {
	return u.name
}

func (u *Unit) Kind() kind.Kind {
	return u.kind
}

// PortoDaemon asserts current spec to PortoDaemon protobuf
func (u *Unit) PortoDaemon() *pb.PortoDaemon {
	return u.spec.(*pb.PortoDaemon)
}

// PackageSet asserts current spec to PackageSetSpec protobuf
func (u *Unit) PackageSet() *pb.PackageSetSpec {
	return u.spec.(*pb.PackageSetSpec)
}

// SystemService asserts current spec to SystemService protobuf
func (u *Unit) SystemService() *pb.SystemServiceSpec {
	return u.spec.(*pb.SystemServiceSpec)
}

// TimerJob asserts current spec to TimerJob protobuf
func (u *Unit) TimerJob() *pb.TimerJobSpec {
	return u.spec.(*pb.TimerJobSpec)
}

// Pod asserts current spec to Pod protobuf
func (u *Unit) Pod() *pb.HostPodSpec {
	return u.spec.(*pb.HostPodSpec)
}

func (u *Unit) SlotMeta() *pb.SlotMeta {
	return u.slotMeta
}

func (u *Unit) RevisionMeta() *pb.RevisionMeta {
	return u.revMeta
}

func (u *Unit) Absent() bool {
	return u.absent
}

func (u *Unit) Prettify() (string, error) {
	return Prettify(u.name, u.digest, u.revMeta, u.slotMeta, u.spec)
}

func Prettify(name, digest string, revMeta *pb.RevisionMeta, slotMeta *pb.SlotMeta, spec proto.Message) (string, error) {
	revMetaYAML, err := yamlutil.ProtoToYaml(revMeta)
	if err != nil {
		return "", fmt.Errorf("failed to marshal meta: %w", err)
	}
	slotMetaYAML, err := yamlutil.ProtoToYaml(slotMeta)
	if err != nil {
		return "", fmt.Errorf("failed to marshal meta: %w", err)
	}
	specYAML, err := yamlutil.ProtoToYaml(spec)
	if err != nil {
		return "", fmt.Errorf("failed to marshal spec: %w", err)
	}
	b := &strings.Builder{}
	// line length
	header := ""
	if len(digest) > 11 {
		header = fmt.Sprintf(" %s %s@rev.%.11s ", name, revMeta.Version, digest)
	}
	b.Write(fmtutil.FmtLine(header))
	b.WriteByte('\n')
	b.Write(fmtutil.FmtLine(" slot meta "))
	b.WriteByte('\n')
	b.Write(slotMetaYAML)
	b.Write(fmtutil.FmtLine(" rev meta "))
	b.WriteByte('\n')
	b.Write(revMetaYAML)
	b.Write(fmtutil.FmtLine(" spec "))
	b.WriteByte('\n')
	b.Write(specYAML)
	b.Write(fmtutil.FmtLine(""))
	b.WriteByte('\n')
	return b.String(), nil
}

func (u *Unit) Valid() error {
	// Skipping absent - we are not interested in spec
	if u.absent {
		return nil
	}
	if err := valid.SlotMeta(u.slotMeta); err != nil {
		return err
	}
	if err := valid.RevisionMeta(u.revMeta); err != nil {
		return err
	}
	switch u.kind {
	case kind.PortoDaemon:
		return valid.PortoDaemon(u.PortoDaemon())
	case kind.PackageSet:
		return valid.PackageSet(u.PackageSet())
	case kind.SystemService:
		return valid.SystemService(u.name, u.SystemService())
	case kind.TimerJob:
		return valid.TimerJob(u.name, u.TimerJob())
	case kind.HostPod:
		return valid.Pod(u.Pod())
	default:
		return fmt.Errorf("unsupported kind: %s", u.kind)
	}
}

func FmtRule(kind kind.Kind, name string) string {
	return fmt.Sprintf("%s-%s", kind.Kebab(), name)
}

func (u *Unit) Throttle(l log.Logger, throttler orly.Throttler, status *pb.SlotStatus) bool {
	l = env.LogWithName(l, fmt.Sprintf("[%s]", u.name))
	rule := FmtRule(u.kind, u.name)
	l.Infof("Checking if %s allows operation...", rule)
	err := throttler.Can(rule, u.slotMeta.Labels)
	if err != nil {
		l.Debugf("Marking '%s' as throttled/pending", u.name)
		pbutil.TrueCond(status.Throttled, err.Error())
		pbutil.TrueCond(status.Pending, err.Error())
		return false
	} else {
		l.Debugf("Marking '%s' as not throttled", u.name)
		pbutil.FalseCond(status.Throttled, "")
		return true
	}
}
