package valid

import (
	"errors"
	"fmt"
	"math"
	"strconv"
	"strings"
	"time"

	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/slices"
)

const (
	minCPUPercent = 0
	maxCPUPercent = 100

	minCPUCores = 0
	maxCPUCores = 30

	// allow long values some packages versions can be long
	// e.g. libasn1-8-heimdal=1.7~git20150920+dfsg-4ubuntu1.16.04.1
	packageNameLen    = 128
	packageVerLen     = 128
	maxPSPackageCount = 500
	maxPackageCount   = 50

	// SystemService retryPolicy validation
	minTimeout = 1 * time.Second
	maxRetries = 5
	// Half of hostctl run
	maxTotalTimeout = 900 * time.Second
)

var (
	envModes = []string{
		"real",   // Actually performs actions
		"noop",   // Does nothing, emulates successful install/start
		"shadow", // Shadow release mode, same as noop, but leaves markers as if run OK
	}
	updatePolicyMethods = []string{
		"restart",
		"reload",
	}
)

func FieldN(f string, min, max int) error {
	if len(f) < min {
		return fmt.Errorf("'%s' is too short (len=%d), min: %d", f, len(f), min)
	}
	if len(f) > max {
		return fmt.Errorf("'%s' is too long (len=%d), max: %d", f, len(f), max)
	}
	return nil
}

func Version(v string) error {
	return FieldN(v, 1, 32)
}

func Stage(s string) error {
	return FieldN(s, 1, 32)
}

func Field64(f string) error {
	return FieldN(f, 1, 64)
}

func PackageVersion(v string) error {
	if len(v) == 0 {
		return errors.New("empty version")
	}
	return FieldN(v, 1, packageVerLen)
}

func PackageName(v string) error {
	if len(v) == 0 {
		return errors.New("empty name")
	}
	return FieldN(v, 1, packageNameLen)
}

func CPUGuarantee(c string) error {
	if len(c) == 0 {
		return nil
	}
	return CPUValue(c, "cpu_guarantee")
}

func CPULimit(c string) error {
	return CPUValue(c, "cpu_limit")
}

// CPUValue validates cpu_guarantee and limit according to porto format.
// Syntax: 0.0..100.0 (in %) | <cores>c (in cores), default: 0 (see porto.md).
func CPUValue(c, field string) error {
	if len(c) == 0 {
		return nil
	}
	// Cpu limit in %
	f, err := strconv.ParseFloat(c, 64)
	if err == nil {
		if math.IsNaN(f) || math.IsInf(f, 0) {
			return fmt.Errorf("%s='%s' is invalid", field, c)
		}
		if f < minCPUPercent {
			return fmt.Errorf("%s='%g' is too small, min=%d", field, f, minCPUPercent)
		}
		if f > maxCPUPercent {
			return fmt.Errorf("%s='%g' is too large, max=%d", field, f, maxCPUPercent)
		}
		return nil
	}
	// Cpu limit in cores
	if c[len(c)-1] != 'c' && c[len(c)-1] != 'C' {
		return fmt.Errorf("invalid %s='%s': %w", field, c, err)
	}
	cores, err := strconv.ParseFloat(c[:len(c)-1], 64)
	if err != nil {
		return fmt.Errorf("invalid %s='%s': %w", field, c, err)
	}
	if cores > maxCPUCores {
		return fmt.Errorf("%s='%s' is too large max='%d'", field, c, maxCPUCores)
	}
	if cores < minCPUCores {
		return fmt.Errorf("%s='%s' is too small min='%d'", field, c, minCPUCores)
	}
	return nil
}

func EnvMode(mode string) error {
	if mode == "" {
		// Alias for 'real' mode
		return nil
	}
	if !slices.ContainsString(envModes, mode) {
		modes := strings.Join(append(envModes, `''`), ", ")
		return fmt.Errorf("invalid value '%s', allowed: [%s]", mode, modes)
	}
	return nil
}

func UpdatePolicy(p *pb.UpdatePolicy) error {
	// allowed for backward capability
	if p == nil {
		return nil
	}
	if err := UpdatePolicyMethod(p.Method); err != nil {
		return fmt.Errorf("update_policy.method: %w", err)
	}
	if p.Retries > maxRetries {
		return fmt.Errorf("update_policy.retries %d is greater than %d", p.Retries, maxRetries)
	}
	totalTimeout := p.Timeout.AsDuration() * time.Duration(p.Retries)
	if totalTimeout > maxTotalTimeout {
		return fmt.Errorf("total (sum of retries) update_policy.timeout %s is greater than %s", totalTimeout.String(), maxTotalTimeout.String())
	}
	return nil
}

func UpdatePolicyMethod(method string) error {
	if method == "" {
		return nil
	}
	if !slices.ContainsString(updatePolicyMethods, method) {
		modes := strings.Join(append(updatePolicyMethods, `''`), ", ")
		return fmt.Errorf("invalid value '%s', allowed: [%s]", method, modes)
	}
	return nil
}

func Files(m []*pb.ManagedFile) error {
	var errs []string
	for i, f := range m {
		if f.Path == "" {
			errs = append(errs, fmt.Sprintf("files[%d].path cannot be empty", i))
		}
	}
	if len(errs) > 0 {
		return fmt.Errorf("%s", strings.Join(errs, ", "))
	} else {
		return nil
	}
}

func PackageSet(m *pb.PackageSetSpec) error {
	// We can have huge package sets, e.g upstream packages.
	if l := len(m.Packages); l > maxPSPackageCount {
		return fmt.Errorf("len(spec.packages)=%d, max=%d", l, maxPSPackageCount)
	}
	if err := Files(m.Files); err != nil {
		return err
	}
	return Packages(m.Packages)
}

func sliceUniq(s []string) bool {
	tmp := make(map[string]bool)
	for _, v := range s {
		if !tmp[v] {
			tmp[v] = true
		} else {
			return false
		}
	}
	return true
}

func sdTemplateValid(name string, template *pb.SystemdTemplate) error {
	if strings.HasSuffix(name, "@") {
		if template == nil {
			return fmt.Errorf("templated unit should have at least one value in spec.template.instances")
		}
		if len(template.Instances) < 1 {
			return fmt.Errorf("templated unit should have at least one value in spec.template.instances")
		}
		if !sliceUniq(template.Instances) {
			return fmt.Errorf("spec.template.instances should contain unique instance names")
		}
	}
	if template != nil && len(template.Instances) > 0 && !strings.HasSuffix(name, "@") {
		return fmt.Errorf("non-templated unit should not have any values in spec.template.instances")
	}
	return nil
}

func SystemService(name string, m *pb.SystemServiceSpec) error {
	if err := sdTemplateValid(name, m.Template); err != nil {
		return err
	}
	if l := len(m.Packages); l > maxPackageCount {
		return fmt.Errorf("len(spec.packages)=%d, max=%d", l, maxPackageCount)
	}
	if err := Packages(m.Packages); err != nil {
		return err
	}
	if err := Files(m.Files); err != nil {
		return err
	}
	return UpdatePolicy(m.UpdatePolicy)
}

func TimerJob(name string, m *pb.TimerJobSpec) error {
	if err := sdTemplateValid(name, m.Template); err != nil {
		return err
	}
	if l := len(m.Packages); l > maxPackageCount {
		return fmt.Errorf("len(spec.packages)=%d, max=%d", l, maxPackageCount)
	}
	if err := Files(m.Files); err != nil {
		return err
	}
	return Packages(m.Packages)
}

func PortoDaemon(m *pb.PortoDaemon) error {
	if l := len(m.Packages); l > maxPackageCount {
		return fmt.Errorf("len(spec.packages)=%d, max=%d", l, maxPackageCount)
	}
	if err := CPUGuarantee(m.Properties.CpuGuarantee); err != nil {
		return fmt.Errorf("spec.properties: %w", err)
	}
	if err := CPULimit(m.Properties.CpuLimit); err != nil {
		return fmt.Errorf("spec.properties: %w", err)
	}
	if err := Files(m.Files); err != nil {
		return err
	}
	return Packages(m.Packages)
}

func Package(m *pb.SystemPackage) error {
	if len(m.Name) == 0 {
		return errors.New("no name provided")
	}
	if err := PackageName(m.Name); err != nil {
		return fmt.Errorf("%s name: %w", m.Name, err)
	}
	if err := PackageVersion(m.Version); err != nil {
		return fmt.Errorf("%s version: %w", m.Name, err)
	}
	return nil
}

func Packages(packages []*pb.SystemPackage) error {
	for i, p := range packages {
		if err := Package(p); err != nil {
			return fmt.Errorf("packages[%d]: %w", i, err)
		}
	}
	return nil
}

func SlotMeta(m *pb.SlotMeta) error {
	if err := Stage(m.Annotations["stage"]); err != nil {
		return fmt.Errorf("meta.annotations['stage']: %w", err)
	}
	if err := EnvMode(m.Annotations["env.mode"]); err != nil {
		return fmt.Errorf("meta.annotations['env.mode']: %w", err)
	}
	for key, val := range m.Annotations {
		if err := Field64(key); err != nil {
			return fmt.Errorf("meta.annotations['%s']: %w", key, err)
		}
		// some fields in annotations could be very long
		// e.g. filename or reconf-juggler
		if err := FieldN(val, 1, 1024); err != nil {
			return fmt.Errorf("meta.annotations['%s']=%s: %w", key, val, err)
		}
	}
	for key, val := range m.Labels {
		if err := Field64(key); err != nil {
			return fmt.Errorf("meta.labels['%s']: %w", key, err)
		}
		if err := Field64(val); err != nil {
			return fmt.Errorf("meta.labels['%s']='%s' %w", key, val, err)
		}
	}
	return nil
}

func RevisionMeta(m *pb.RevisionMeta) error {
	if err := Version(m.Version); err != nil {
		return fmt.Errorf("meta.version: %w", err)
	}
	return nil
}

func Pod(pod *pb.HostPodSpec) error {
	var errs []error
	if err := Packages(pod.Packages); err != nil {
		errs = append(errs, err)
	}
	if err := Files(pod.Files); err != nil {
		errs = append(errs, err)
	}
	for i, d := range pod.PortoDaemons {
		pd := &pb.PortoDaemon{
			Properties: d.Properties,
		}
		if err := PortoDaemon(pd); err != nil {
			errs = append(errs, fmt.Errorf("porto_daemons[%d]: %v", i, err))
		}
	}
	for i, s := range pod.Services {
		ss := &pb.SystemServiceSpec{
			UpdatePolicy: s.UpdatePolicy,
			Template:     s.Template,
		}
		if err := SystemService(s.Name, ss); err != nil {
			errs = append(errs, fmt.Errorf("services[%d]: %v", i, err))
		}
	}
	for i, t := range pod.Timers {
		tj := &pb.TimerJobSpec{
			Template: t.Template,
		}
		if err := TimerJob(t.Name, tj); err != nil {
			errs = append(errs, fmt.Errorf("timers[%d]: %v", i, err))
		}
	}
	if len(errs) > 0 {
		errStrings := make([]string, len(errs))
		for i, err := range errs {
			errStrings[i] = err.Error()
		}
		return fmt.Errorf("errors validating pod: %s", strings.Join(errStrings, ", "))
	}
	return nil
}
