package valid

import (
	"errors"
	"fmt"
	"math/rand"
	"strconv"
	"strings"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"google.golang.org/protobuf/types/known/durationpb"

	"a.yandex-team.ru/infra/hostctl/pkg/proptest"
	pb "a.yandex-team.ru/infra/hostctl/proto"
)

func TestSlotMeta(t *testing.T) {
	type initFun func(meta *pb.SlotMeta) // Init meta object before test case
	type testCase struct {
		name string
		init initFun
		err  string
	}
	tests := []testCase{
		{
			name: "too long stage",
			init: func(meta *pb.SlotMeta) {
				meta.Annotations["stage"] = strings.Repeat("x", 33)
			},
			err: "meta.annotations['stage']: 'xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx' " +
				"is too long (len=33), max: 32",
		},
		{
			name: "invalid mode",
			init: func(meta *pb.SlotMeta) {
				meta.Annotations["stage"] = "default"
				meta.Annotations["env.mode"] = "surreal"
			},
			err: "meta.annotations['env.mode']: invalid value 'surreal', " +
				"allowed: [real, noop, shadow, '']",
		},
		{
			name: "invalid annotation key",
			init: func(meta *pb.SlotMeta) {
				meta.Annotations["stage"] = "default"
				meta.Annotations["env.mode"] = "real"
				meta.Annotations[strings.Repeat("y", 65)] = "a"
			},
			err: "meta.annotations['yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy']: " +
				"'yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy' " +
				"is too long (len=65), max: 64",
		},
		{
			name: "invalid annotation value",
			init: func(meta *pb.SlotMeta) {
				meta.Annotations["stage"] = "default"
				meta.Annotations["env.mode"] = "real"
				meta.Annotations["some"] = strings.Repeat("y", 1025)
			},
			err: fmt.Sprintf("meta.annotations['some']=%s: '%s' is too long (len=1025), max: 1024", strings.Repeat("y", 1025), strings.Repeat("y", 1025)),
		},
		{
			name: "invalid label key",
			init: func(meta *pb.SlotMeta) {
				meta.Annotations["stage"] = "default"
				meta.Labels[strings.Repeat("y", 65)] = "a"
			},
			err: "meta.labels['yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy']: " +
				"'yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy' " +
				"is too long (len=65), max: 64",
		},
		{
			name: "invalid label value",
			init: func(meta *pb.SlotMeta) {
				meta.Annotations["stage"] = "default"
				meta.Labels["some"] = strings.Repeat("y", 65)
			},
			err: "meta.labels['some']='yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy' " +
				"'yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy' " +
				"is too long (len=65), max: 64",
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			m := &pb.SlotMeta{
				Labels:      make(map[string]string),
				Annotations: make(map[string]string),
			}
			tt.init(m)
			err := SlotMeta(m)
			if tt.err == "" {
				assert.NoError(t, err)
			} else {
				assert.EqualError(t, err, tt.err)
			}
		})
	}
}

func TestRevisionMeta(t *testing.T) {
	type initFun func(meta *pb.RevisionMeta) // Init meta object before test case
	type testCase struct {
		name string
		init initFun
		err  string
	}
	tests := []testCase{
		{
			name: "empty version",
			init: func(meta *pb.RevisionMeta) {
				meta.Version = ""
			},
			err: "meta.version: '' is too short (len=0), min: 1",
		},
		{
			name: "too long version",
			init: func(meta *pb.RevisionMeta) {
				meta.Version = strings.Repeat("1", 33)
			},
			err: "meta.version: '111111111111111111111111111111111' " +
				"is too long (len=33), max: 32",
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			m := &pb.RevisionMeta{}
			tt.init(m)
			err := RevisionMeta(m)
			if tt.err == "" {
				assert.NoError(t, err)
			} else {
				assert.EqualError(t, err, tt.err)
			}
		})
	}
}

func TestCPUValue(t *testing.T) {
	tests := []struct {
		name string
		val  string
		err  string
	}{
		{
			name: "empty",
			val:  "",
		},
		{
			name: "negative",
			val:  "-43",
			err:  "cpu_guarantee='-43' is too small, min=0",
		},
		{
			name: "NaN",
			val:  "NaN",
			err:  "cpu_guarantee='NaN' is invalid",
		},
		{
			name: "inf",
			val:  "inf",
			err:  "cpu_guarantee='inf' is invalid",
		},
		{
			name: "too many %",
			val:  "146",
			err:  "cpu_guarantee='146' is too large, max=100",
		},
		{
			name: "invalid non percent",
			val:  "14d",
			err:  `invalid cpu_guarantee='14d': strconv.ParseFloat: parsing "14d": invalid syntax`,
		},
		{
			name: "invalid cores",
			val:  "13fc",
			err:  "invalid cpu_guarantee='13fc': strconv.ParseFloat: parsing \"13f\": invalid syntax",
		},
		{
			name: "too small cores",
			val:  "-0.0001c",
			err:  "cpu_guarantee='-0.0001c' is too small min='0'",
		},
		{
			name: "too many cores",
			val:  "256c",
			err:  "cpu_guarantee='256c' is too large max='30'",
		},
		{
			name: "ok percent",
			val:  "16",
		},
		{
			name: "ok cores",
			val:  "16c",
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			err := CPUValue(tt.val, "cpu_guarantee")
			if len(tt.err) != 0 {
				if err == nil {
					t.Errorf("no error, but want some")
				} else if tt.err != err.Error() {
					t.Errorf("error='%s', but want '%s'", err.Error(), tt.err)
				}
			} else {
				if err != nil {
					t.Errorf("unexpected error: %s", err.Error())
				}
			}
		})
	}
}

func TestPackage(t *testing.T) {
	type initFun func(m *pb.SystemPackage) // Init meta object before test case
	type testCase struct {
		name string
		init initFun
		err  string
	}
	tests := []testCase{
		{
			name: "no name",
			init: func(m *pb.SystemPackage) {
				m.Version = "1.0"
			},
			err: "no name provided",
		},
		{
			name: "no version",
			init: func(m *pb.SystemPackage) {
				m.Name = "yandex-hostctl"
			},
			err: "yandex-hostctl version: empty version",
		},
		{
			name: "long version",
			init: func(m *pb.SystemPackage) {
				m.Name = "yandex-hostctl"
				m.Version = strings.Repeat("1", 129)
			},
			err: fmt.Sprintf("yandex-hostctl version: '%s' is too long (len=129), max: 128", strings.Repeat("1", 129)),
		},
		{
			name: "okay",
			init: func(m *pb.SystemPackage) {
				m.Name = "yandex-hostctl"
				m.Version = "1.0-181615651"
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			m := &pb.SystemPackage{}
			tt.init(m)
			err := Package(m)
			if tt.err == "" {
				assert.NoError(t, err)
			} else {
				assert.EqualError(t, err, tt.err)
			}
		})
	}
}

// Simple test to check error message ux/readability.
func TestPackages(t *testing.T) {
	m := []*pb.SystemPackage{
		{
			Name:    "hostctl",
			Version: "1.0",
		},
		{
			Name:    "hostman",
			Version: "42.0",
		},
		{
			Name:    "saltstack",
			Version: "",
		},
	}
	err := Packages(m)
	assert.EqualError(t, err, "packages[2]: saltstack version: empty version")
}

func TestUpdatePolicy(t *testing.T) {
	reload := &pb.UpdatePolicy{
		Method: "reload",
	}
	invalid := &pb.UpdatePolicy{
		Method: "invalid",
	}
	var empty *pb.UpdatePolicy
	err := UpdatePolicy(reload)
	assert.NoError(t, err)
	err = UpdatePolicy(empty)
	assert.NoError(t, err)
	err = UpdatePolicy(invalid)
	assert.EqualError(t, err, "update_policy.method: invalid value 'invalid', allowed: [restart, reload, '']")
	retriesOverLimit := &pb.UpdatePolicy{Retries: 6}
	err = UpdatePolicy(retriesOverLimit)
	assert.EqualError(t, err, "update_policy.retries 6 is greater than 5")
	timeoutOverflow := &pb.UpdatePolicy{Timeout: durationpb.New(301 * time.Second), Retries: 3}
	err = UpdatePolicy(timeoutOverflow)
	assert.EqualError(t, err, "total (sum of retries) update_policy.timeout 15m3s is greater than 15m0s")
	reload.Retries = 2
	reload.Timeout = durationpb.New(time.Minute)
	assert.NoError(t, UpdatePolicy(reload))
}

func TestPortoDaemon(t *testing.T) {
	// Simple package count test
	m := &pb.PortoDaemon{}
	for i := 0; i < 51; i++ {
		m.Packages = append(m.Packages, &pb.SystemPackage{
			Name:    strconv.Itoa(i),
			Version: strconv.Itoa(i + 100),
		})
	}
	err := PortoDaemon(m)
	assert.EqualError(t, err, "len(spec.packages)=51, max=50")
}

func TestPackageSet(t *testing.T) {
	// Simple package count test
	m := &pb.PackageSetSpec{}
	for i := 0; i < 501; i++ {
		m.Packages = append(m.Packages, &pb.SystemPackage{
			Name:    strconv.Itoa(i),
			Version: strconv.Itoa(i + 100),
		})
	}
	err := PackageSet(m)
	assert.EqualError(t, err, "len(spec.packages)=501, max=500")
}

func TestSystemService(t *testing.T) {
	// Simple package count test
	m := &pb.SystemServiceSpec{}
	for i := 0; i < 51; i++ {
		m.Packages = append(m.Packages, &pb.SystemPackage{
			Name:    strconv.Itoa(i),
			Version: strconv.Itoa(i + 100),
		})
	}
	err := SystemService("mock", m)
	assert.EqualError(t, err, "len(spec.packages)=51, max=50")
}

func TestTimerJob(t *testing.T) {
	// Simple package count test
	m := &pb.TimerJobSpec{}
	for i := 0; i < 51; i++ {
		m.Packages = append(m.Packages, &pb.SystemPackage{
			Name:    strconv.Itoa(i),
			Version: strconv.Itoa(i + 100),
		})
	}
	err := TimerJob("mock", m)
	assert.EqualError(t, err, "len(spec.packages)=51, max=50")
}

func TestPackageVersion(t *testing.T) {
	// persistence random source for reproducibility tests
	err := proptest.Check(
		func(r *rand.Rand) interface{} {
			// gen valid and invalid strings
			return proptest.RandString(r, 0, 256)
		},
		func(args ...interface{}) error {
			m := &pb.SystemPackage{
				Name:    "yandex-hostctl",
				Version: args[0].(string),
			}
			return Package(m)
		},
		func(args ...interface{}) error {
			version := args[0].(string)
			if len(version) < 1 {
				return fmt.Errorf("yandex-hostctl version: empty version")
			}
			if len(version) > 128 {
				return fmt.Errorf("yandex-hostctl version: '%s' is too long (len=%d), max: 128", version, len(version))
			}
			return nil
		},
	)
	if err != nil {
		t.Errorf("invalid validation: %s", err)
	}
}

func TestPackageName(t *testing.T) {
	// persistence random source for reproducibility tests
	err := proptest.Check(
		func(r *rand.Rand) interface{} {
			// gen valid and invalid strings
			return proptest.RandString(r, 0, 256)
		},
		func(args ...interface{}) error {
			m := &pb.SystemPackage{
				Name:    args[0].(string),
				Version: "v1",
			}
			return Package(m)
		},
		func(args ...interface{}) error {
			name := args[0].(string)
			if len(name) < 1 {
				return errors.New("no name provided")
			}
			if len(name) > 128 {
				return fmt.Errorf("%s name: '%s' is too long (len=%d), max: 128", name, name, len(name))
			}
			return nil
		},
	)
	if err != nil {
		t.Errorf("invalid validation: %s", err)
	}
}

func TestSliceUniq(t *testing.T) {
	assert.False(t, sliceUniq([]string{"mock", "mock"}))
	assert.True(t, sliceUniq([]string{"mock1", "mock2"}))
}

func TestSDTemplateValid(t *testing.T) {
	template := pb.SystemdTemplate{}
	assert.NoError(t, sdTemplateValid("mock", &template))
	assert.EqualError(t, sdTemplateValid("mock@", nil), "templated unit should have at least one value in spec.template.instances")
	assert.EqualError(t, sdTemplateValid("mock@", &template), "templated unit should have at least one value in spec.template.instances")
	template.Instances = append(template.Instances, "mock")
	assert.EqualError(t, sdTemplateValid("mock", &template), "non-templated unit should not have any values in spec.template.instances")
	assert.NoError(t, sdTemplateValid("mock@", &template))
	template.Instances = append(template.Instances, "mock2")
	assert.NoError(t, sdTemplateValid("mock@", &template))
	assert.NoError(t, sdTemplateValid("mock", nil))
	template.Instances = append(template.Instances, "mock2")
	assert.EqualError(t, sdTemplateValid("mock@", &template), "spec.template.instances should contain unique instance names")
}

func TestFiles(t *testing.T) {
	m := make([]*pb.ManagedFile, 0)
	assert.NoError(t, Files(m))
	m = append(m, &pb.ManagedFile{Path: "/dev/null"})
	assert.NoError(t, Files(m))
	m = append(m, &pb.ManagedFile{Content: "/dev/zero"})
	assert.EqualError(t, Files(m), "files[1].path cannot be empty")
}

func TestPodValid_Ok(t *testing.T) {
	spec := &pb.HostPodSpec{
		Packages: []*pb.SystemPackage{
			{
				Name:    "test-package",
				Version: "test-ver",
			},
		},
		Files: []*pb.ManagedFile{
			{
				Path:    "/test-file",
				Content: "test-content",
				User:    "mock",
				Group:   "mock",
				Mode:    "666",
			},
		},
		PortoDaemons: []*pb.HostPodPortoDaemon{
			{
				Name: "test-pd",
				Properties: &pb.PortoProperties{
					Cmd: []string{"/test-cmd"},
				},
			},
		},
		Services: []*pb.HostPodService{
			{
				Name:         "test-service",
				UpdatePolicy: nil,
				Template:     nil,
			},
		},
		Timers: []*pb.HostPodTimer{
			{
				Name: "test-timer",
			},
		},
	}
	assert.NoError(t, Pod(spec))
}

func TestPodValid_PkgErrors(t *testing.T) {
	spec := &pb.HostPodSpec{
		Packages: []*pb.SystemPackage{
			{
				Version: "test-ver",
			},
		},
	}
	assert.EqualError(t, Pod(spec), "errors validating pod: packages[0]: no name provided")
}

func TestPodValid_FileErrors(t *testing.T) {
	spec := &pb.HostPodSpec{
		Files: []*pb.ManagedFile{
			{
				Content: "mock",
			},
		},
	}
	assert.EqualError(t, Pod(spec), "errors validating pod: files[0].path cannot be empty")
}

func TestPodValid_PDErrors(t *testing.T) {
	spec := &pb.HostPodSpec{
		PortoDaemons: []*pb.HostPodPortoDaemon{
			{
				Name: "test-pd",
				Properties: &pb.PortoProperties{
					Cmd:      []string{"/test-cmd"},
					CpuLimit: "-1GPU",
				},
			},
		},
	}
	assert.EqualError(t, Pod(spec), "errors validating pod: porto_daemons[0]: spec.properties: invalid cpu_limit='-1GPU': strconv.ParseFloat: parsing \"-1GPU\": invalid syntax")
}

func TestPodValid_SDErrors(t *testing.T) {
	spec := &pb.HostPodSpec{
		Services: []*pb.HostPodService{
			{
				Name:     "test@",
				Template: &pb.SystemdTemplate{},
			},
		},
	}
	assert.EqualError(t, Pod(spec), "errors validating pod: services[0]: templated unit should have at least one value in spec.template.instances")
}

func TestPodValid_TJErrors(t *testing.T) {
	spec := &pb.HostPodSpec{
		Timers: []*pb.HostPodTimer{
			{
				Name:     "test@",
				Template: &pb.SystemdTemplate{},
			},
		},
	}
	assert.EqualError(t, Pod(spec), "errors validating pod: timers[0]: templated unit should have at least one value in spec.template.instances")
}
