package env

import (
	"a.yandex-team.ru/infra/hostctl/internal/orly"
	"a.yandex-team.ru/infra/hostctl/internal/systemd"
	"a.yandex-team.ru/infra/hostctl/internal/systemd/persist"
	"a.yandex-team.ru/infra/hostctl/internal/units/env/file"
	"a.yandex-team.ru/infra/hostctl/internal/units/env/pacman"
	"a.yandex-team.ru/infra/hostctl/internal/units/env/porto"
	"a.yandex-team.ru/infra/hostctl/internal/units/env/sleep"
	"a.yandex-team.ru/library/go/core/log"
	"fmt"
)

type Mode string

const (
	ShadowMode = "shadow"
	NoopMode   = "noop"
	RealMode   = "real"
	TestMode   = "test"
)

type Env struct {
	Mode        string
	L           log.Logger
	Pacman      pacman.PackageManager
	FileBuilder file.Builder
	Throttler   orly.Throttler
	Porto       porto.Porto
	Systemd     systemd.Systemd
	Sleep       sleep.Sleep
}

func LogWithName(l log.Logger, name string) log.Logger {
	return l.WithName(name)
}

func LogWithNoop(l log.Logger) log.Logger {
	return LogWithName(l, NoopMode)
}

func LogWithShadow(l log.Logger) log.Logger {
	return LogWithName(l, ShadowMode)
}

func (e *Env) For(name string) *Env {
	return &Env{
		L:           LogWithName(e.L, fmt.Sprintf("[%s]", name)),
		Pacman:      e.Pacman,
		FileBuilder: e.FileBuilder,
		Throttler:   e.Throttler,
		Porto:       e.Porto,
		Systemd:     e.Systemd,
		Sleep:       e.Sleep,
		Mode:        e.Mode,
	}
}

func Noop(l log.Logger) *Env {
	l = LogWithNoop(l)
	return &Env{
		L:           l,
		Pacman:      pacman.NewNoop(pacman.NewDpkg(l)),
		FileBuilder: file.NewNoopFile,
		Throttler:   orly.NewNoop(),
		Porto:       porto.NewNoopPorto(),
		Systemd:     systemd.NewNoopSystemd(systemd.NewSystemctl(persist.NewNoop(), l)),
		Sleep:       sleep.Noop,
		Mode:        NoopMode,
	}
}

func Real(l log.Logger, noorly bool, orlyURL, hostname string) *Env {
	var throttler orly.Throttler
	if noorly {
		throttler = orly.NewNoop()
	} else {
		throttler = orly.NewOrly(hostname, orlyURL)
	}
	return &Env{
		L:           l,
		FileBuilder: file.NewRealFile,
		Pacman:      pacman.NewDpkg(l),
		Throttler:   throttler,
		Porto:       porto.NewProduction(),
		// we use systemctl subprocess implementation because we had a lot of problems
		// with go-systemd lib / dbus calls (magic listener here https://github.com/coreos/go-systemd/blob/v22.1.0/dbus/dbus.go#L186),
		// systemctl thick client (that calls SysV script for systemctl enable/disable/is-enabled <SysV unit>)
		Systemd: systemd.NewSystemctl(persist.NewReal(), l),
		Sleep:   sleep.Real,
		Mode:    RealMode,
	}
}

func Shadow(l log.Logger) *Env {
	l = LogWithShadow(l)
	return &Env{
		L:           l,
		Pacman:      pacman.NewNoop(pacman.NewDpkg(l)),
		FileBuilder: file.NewNoopFile,
		Throttler:   orly.NewNoop(),
		Porto:       porto.NewNoopPorto(),
		Systemd:     systemd.NewShadowSystemctl(systemd.NewSystemctl(persist.NewReal(), l), persist.NewReal()),
		Sleep:       sleep.Noop,
		Mode:        ShadowMode,
	}
}
