package file

import (
	"fmt"
	"io/ioutil"
	"os"
	"os/user"
	"strconv"
	"syscall"

	"github.com/pmezard/go-difflib/difflib"

	pb "a.yandex-team.ru/infra/hostctl/proto"
)

type Builder func(path, content, user, group, mode string) File

type File interface {
	Manage() error
	Remove() error
	Path() string
	Content() string
	User() string
	Group() string
	Mode() string
	FromFs() (File, error)
}

func Changes(a, b File) map[string]string {
	diff := make(map[string]string)
	if a.Content() != b.Content() {
		d := difflib.UnifiedDiff{
			A:        difflib.SplitLines(a.Content()),
			B:        difflib.SplitLines(b.Content()),
			FromFile: "before",
			ToFile:   "after",
			Context:  3,
		}
		diff["content"], _ = difflib.GetUnifiedDiffString(d)
	}
	if a.Mode() != b.Mode() {
		diff["mode"] = fmt.Sprintf("%s -> %s", a.Mode(), b.Mode())
	}
	if a.User() != b.User() {
		diff["user"] = fmt.Sprintf("%s -> %s", a.User(), b.User())
	}
	if a.Group() != b.Group() {
		diff["group"] = fmt.Sprintf("%s -> %s", a.Group(), b.Group())
	}
	if len(diff) != 0 {
		return diff
	}
	return nil
}

// Noent initializes non-existing managed file factory.
func Noent(path string) File {
	return newFile(NewNoopFile, path, "", "", "", "")
}

// FromPb initializes file object from protobuf counterpart.
func FromPb(builder Builder, file *pb.ManagedFile) File {
	u := file.User
	if u == "" {
		u = "root"
	}
	g := file.Group
	if g == "" {
		g = "root"
	}
	mode := file.Mode
	if mode == "" {
		mode = "644"
	}
	return newFile(builder, file.Path, file.Content, u, g, mode)
}

// FromFs initialize managed file from filesystem and os.
func FromFs(builder Builder, path string) (File, error) {
	f, err := os.Open(path)
	if err != nil {
		if os.IsNotExist(err) {
			return Noent(path), nil
		}
		return nil, err
	}
	defer f.Close()
	fstat := &syscall.Stat_t{}
	err = syscall.Fstat(int(f.Fd()), fstat)
	if err != nil {
		return nil, fmt.Errorf("fstat '%s' failed: %w", path, err)
	}
	mode := os.FileMode(fstat.Mode)
	if mode.IsDir() {
		return nil, fmt.Errorf("failed to read '%s' - is a directory, something went wrong", f.Name())
	}
	// 420 -> 0644
	modeStr := strconv.FormatUint(uint64(mode.Perm()), 8)
	u, err := user.LookupId(strconv.FormatUint(uint64(fstat.Uid), 10))
	if err != nil {
		return nil, fmt.Errorf("lookup uid '%d' failed: %w", fstat.Uid, err)
	}
	g, err := user.LookupGroupId(strconv.FormatUint(uint64(fstat.Gid), 10))
	if err != nil {
		return nil, fmt.Errorf("lookup gid '%d' failed: %w", fstat.Gid, err)
	}
	content, err := ioutil.ReadAll(f)
	if err != nil {
		return nil, fmt.Errorf("read file '%s' failed: %w", path, err)
	}
	return newFile(builder, path, string(content), u.Username, g.Name, modeStr), nil
}

func newFile(builder Builder, path, content, user, group, mode string) File {
	// check if file mode defined in octal e.g. as '0664'
	if len(mode) == 4 && mode[0] == '0' {
		mode = mode[1:]
	}
	return builder(path, content, user, group, mode)
}
