package file

import (
	"errors"
	"fmt"
	"os"
	"os/user"
	"path/filepath"
	"strconv"
	"syscall"

	"a.yandex-team.ru/infra/hostctl/internal/fileutil"
)

func NewRealFile(path, content, user, group, mode string) File {
	return &RealFile{path, content, user, group, mode}
}

// RealFile object which:
//  * can be managed (provided Content and attributes)
type RealFile struct {
	path    string
	content string
	user    string
	group   string
	mode    string
}

func (f *RealFile) Path() string {
	return f.path
}
func (f *RealFile) Content() string {
	return f.content
}
func (f *RealFile) User() string {
	return f.user
}
func (f *RealFile) Group() string {
	return f.group
}
func (f *RealFile) Mode() string {
	return f.mode
}

func (f *RealFile) Manage() error {
	mode, err := strconv.ParseUint(f.Mode(), 8, 32)
	if err != nil {
		return err
	}
	// Create directories if needed
	dirName := filepath.Dir(f.Path())
	if dirName != "/" {
		err := os.MkdirAll(dirName, os.ModePerm)
		if err != nil && err != syscall.EEXIST {
			return fmt.Errorf("failed to create %s: %w", dirName, err)
		}
	}
	u, err := user.Lookup(f.User())
	if err != nil {
		return fmt.Errorf("user '%s' lookup failed: %w", f.User(), err)
	}
	uid, err := strconv.ParseUint(u.Uid, 10, 32)
	if err != nil {
		return fmt.Errorf("can not convert uid '%s' to int: %w", u.Uid, err)
	}
	g, err := user.LookupGroup(f.Group())
	if err != nil {
		return fmt.Errorf("group '%s' lookup failed: %w", f.Group(), err)
	}
	gid, err := strconv.ParseUint(g.Gid, 10, 32)
	if err != nil {
		return fmt.Errorf("can not convert gid '%s' to int: %w", g.Gid, err)
	}
	return fileutil.AtomicWrite(f.Path(), []byte(f.Content()), os.FileMode(mode), true, true, int(uid), int(gid))
}

func (f *RealFile) Remove() error {
	err := syscall.Unlink(f.path)
	if err != nil {
		if errors.Is(err, syscall.EISDIR) {
			return fmt.Errorf("failed to remove '%s' - is a directory, something went wrong, will not remove directory", f.path)
		}
		return err
	}
	return nil
}

func (f *RealFile) FromFs() (File, error) {
	return FromFs(NewRealFile, f.Path())
}
