package pacman

import (
	"fmt"
	"os"
	"strings"
	"time"

	"a.yandex-team.ru/infra/hostctl/internal/executil"
	"a.yandex-team.ru/infra/hostctl/internal/units/env/pacman/dpkgutil"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
)

var (
	dpkgFix       = []string{"/usr/bin/dpkg", "--configure", "-a"}
	aptFix        = []string{"/usr/bin/apt", "install", "-f"}
	installPrefix = []string{"/usr/bin/apt-get",
		"install",
		"-q", "-y",
		// These options are taken from salt for compatibility
		"-o", "DPkg::Options::=--force-confold",
		"-o", "DPkg::Options::=--force-confdef",
		// 12.04 doesn't have --allow-downgrade, so for now
		// let's live with this one:
		"--force-yes",
	}
	purgePrefix = []string{"/usr/bin/apt-get", "purge", "-q", "-y"}
	aptUpdate   = []string{"/usr/bin/apt-get", "-qq", "update"}
)

func NewDpkg(l log.Logger) *Dpkg {
	// TODO: [6/10/20] (vaspahomov): rewire it later :)
	metrics := &metrics{
		installedOk:     unistat.NewNumeric("hostctl-install-ok", 10, aggr.Counter(), unistat.Sum),
		installFailures: unistat.NewNumeric("hostctl-install-fail", 10, aggr.Counter(), unistat.Sum),
		fixAttempts:     unistat.NewNumeric("hostctl-fix-attempts", 10, aggr.Counter(), unistat.Sum),
		fixFailures:     unistat.NewNumeric("hostctl-fix-fail", 10, aggr.Counter(), unistat.Sum),
		purgedOk:        unistat.NewNumeric("hostctl-purge-ok", 10, aggr.Counter(), unistat.Sum),
		purgeFailures:   unistat.NewNumeric("hostctl-purged-fail", 10, aggr.Counter(), unistat.Sum),
	}
	cmds := &DpkgCmds{
		executil.NewArgv(dpkgFix...),
		executil.NewArgv(aptFix...),
		executil.NewArgv(installPrefix...),
		executil.NewArgv(purgePrefix...),
		executil.NewArgv(aptUpdate...),
	}
	return &Dpkg{metrics, l, cmds}
}

type Dpkg struct {
	*metrics
	l    log.Logger
	cmds *DpkgCmds
}

type DpkgCmds struct {
	DpkgFix       *executil.Cmd
	AptFix        *executil.Cmd
	InstallPrefix *executil.Cmd
	PurgePrefix   *executil.Cmd
	AptGetUpdate  *executil.Cmd
}

func (dpkg *Dpkg) getEnv() ([]string, error) {
	// Default TMPDIR is currently pointing to /place/vartmp, which is typically
	// consumed by user containers, thus free space can be exhausted.
	// Unfortunately apt saves some files in TMPDIR.
	// To make salt more robust, let's try /tmp too.
	origTmpDir := os.Getenv("TMPDIR")
	tmpDir, err := getTmpDir(origTmpDir)
	if err != nil {
		return nil, fmt.Errorf("failed to get tmp dir: %w", err)
	}
	// tmpDir can be empty if there is no space left, but let's try
	// maybe apt will work
	env := executil.Environ(DpkgEnvVars)
	if tmpDir != "" {
		env = append(env, "TMPDIR="+tmpDir)
		if tmpDir != origTmpDir && origTmpDir != "" {
			dpkg.l.Infof("TMPDIR overridden %s -> %s", origTmpDir, tmpDir)
		}
	}
	return env, nil
}

// Tries to fix dpkg by running what it suggests.
func (dpkg *Dpkg) Repair() (bool, error) {
	dpkg.fixAttempts.Update(1)
	dpkg.l.Debugf("Fixing dpkg with %s...", dpkg.cmds.DpkgFix.String())
	_, _, status := executil.CheckOutput(dpkg.cmds.DpkgFix, DpkgFixTimeout,
		executil.Environ(DpkgEnvVars))
	if !status.Ok {
		dpkg.l.Warnf("Fix failed: %s", status.Message)
	} else {
		dpkg.l.Infof("Fix command finished successfully!")
		// Seems there is no need for second action
		return true, nil
	}
	dpkg.l.Debugf("Fixing with %s...", dpkg.cmds.AptFix.String())
	_, _, status = executil.CheckOutput(dpkg.cmds.AptFix, DpkgFixTimeout, executil.Environ(DpkgEnvVars))
	if !status.Ok {
		dpkg.fixFailures.Update(1)
		dpkg.l.Errorf("Fix failed: %s", status.Message)
	} else {
		dpkg.l.Infof("Fix command finished successfully!")
	}
	return status.Ok, nil
}

//Runs package management command, trying to fix and restart it if needed.
func (dpkg *Dpkg) runDpkgCommand(cmdline *executil.Cmd, timeout time.Duration, env []string) ([]byte, []byte, *executil.Status) {
	if env == nil {
		var err error
		env, err = dpkg.getEnv()
		if err != nil {
			return nil, nil, &executil.Status{Ok: false, Message: fmt.Sprintf("failed to get env: %s", err)}
		}
	}
	dpkg.l.Debugf("dpkg command: %s...", cmdline.String())
	stdout, stderr, status := executil.CheckOutput(cmdline, timeout, env)
	if !status.Ok && dpkgNeedsFix(stderr) {
		dpkg.l.Warnf("Execution failed: %s...", stderr)
		dpkg.l.Infof("Trying to fix package...")
		ok, err := dpkg.Repair()
		if err != nil {
			dpkg.l.Errorf("Fix failed...")
			return nil, nil, &executil.Status{
				Ok:      false,
				Message: "Dpkg fix failed",
			}
		}
		if ok {
			dpkg.l.Infof("Fixing finished with success!")
			return executil.CheckOutput(cmdline, timeout, env)
		}
	}
	dpkg.l.Debug("Execution finished successfully")
	dpkg.l.Debugf("dpkg output: %s", stdout)
	return stdout, stderr, status
}

func (dpkg *Dpkg) Update() error {
	env, err := dpkg.getEnv()
	if err != nil {
		return fmt.Errorf("failed to get env: %w", err)
	}
	dpkg.l.Infof("Running %s...", dpkg.cmds.AptGetUpdate.String())
	_, stderr, status := dpkg.runDpkgCommand(dpkg.cmds.AptGetUpdate, AptGetTimeout, env)
	if !status.Ok {
		return fmt.Errorf(strings.Join([]string{status.Message, executil.TailOf(stderr, 100, "stderr: ")}, "; "))
	}
	return nil
}

func (dpkg *Dpkg) GetPackageStatus(pName string) (*dpkgutil.PackageStatus, error) {
	return dpkgutil.ReadStatus(pName)
}

func (dpkg *Dpkg) List(names []string) (map[string]*dpkgutil.PackageStatus, error) {
	rv := make(map[string]*dpkgutil.PackageStatus, len(names))
	for _, name := range names {
		rv[name] = &dpkgutil.PackageStatus{
			Installed: false,
			Name:      name,
			Version:   "unknown",
		}
	}
	if err := dpkgutil.ReadStatuses(rv); err != nil {
		return nil, err
	}
	return rv, nil
}

func (dpkg *Dpkg) Install(name string, version string) (*dpkgutil.PackageStatus, error) {
	return dpkg.install(name, version, false)
}

func (dpkg *Dpkg) InstallDryRun(name string, version string) (*dpkgutil.PackageStatus, error) {
	return dpkg.install(name, version, true)
}

func (dpkg *Dpkg) install(name string, version string, dryRun bool) (*dpkgutil.PackageStatus, error) {
	pkgStr := fmt.Sprintf("%s=%s", name, version)
	cmd := dpkg.cmds.InstallPrefix
	if dryRun {
		cmd = cmd.AddArg("--dry-run")
	}
	cmd = cmd.AddArg(pkgStr)
	stdout, stderr, status := dpkg.runDpkgCommand(cmd, InstallTimeout, nil)
	if !status.Ok {
		dpkg.installFailures.Update(1)
		return nil, fmt.Errorf("failed to install %s: %s\nstdout:\n%s\nstderr:\n%s", pkgStr, status.Message, stdout, stderr)
	}
	dpkg.installedOk.Update(1)
	return &dpkgutil.PackageStatus{
		Name:      name,
		Version:   version,
		Installed: true}, nil
}

func (dpkg *Dpkg) InstallSet(packageSet []Package) error {
	return dpkg.installSet(packageSet, false)
}

func (dpkg *Dpkg) InstallSetDryRun(packageSet []Package) error {
	return dpkg.installSet(packageSet, true)
}

func (dpkg *Dpkg) installSet(packageSet []Package, dryRun bool) error {
	cmd := dpkg.cmds.InstallPrefix
	if dryRun {
		cmd = cmd.AddArg("--dry-run")
	}
	n := 0
	for _, pkg := range packageSet {
		pkgStr := fmt.Sprintf("%s=%s", pkg.Name, pkg.Version)
		cmd = cmd.AddArg(pkgStr)
		n += 1
	}
	stdout, stderr, status := dpkg.runDpkgCommand(cmd, SetTimeout, nil)
	if !status.Ok {
		dpkg.installFailures.Update(float64(n))
		return fmt.Errorf("failed to install packages: %s\nstdout:\n%s\nstderr:\n%s", status.Message, stdout, stderr)
	}
	dpkg.installedOk.Update(float64(n))
	return nil
}

func (dpkg *Dpkg) Purge(pName string) error {
	cmd := dpkg.cmds.PurgePrefix.AddArg(pName)
	stdout, stderr, status := dpkg.runDpkgCommand(cmd, PurgeTimeout, nil)
	if !status.Ok {
		dpkg.purgeFailures.Update(1)
		return fmt.Errorf("failed to purge %s: %s\nstdout:\n%s\nstderr:\n%s", pName, status.Message, stdout, stderr)
	}
	dpkg.purgedOk.Update(1)
	return nil
}

func (dpkg *Dpkg) PurgeSet(names []string) error {
	cmd := dpkg.cmds.PurgePrefix.AddArg(names...)
	stdout, stderr, status := dpkg.runDpkgCommand(cmd, PurgeTimeout, nil)
	if !status.Ok {
		dpkg.purgeFailures.Update(1)
		return fmt.Errorf("failed to purge %s: %s\nstdout:\n%s\nstderr:\n%s", strings.Join(names, ", "), status.Message, stdout, stderr)
	}
	dpkg.purgedOk.Update(1)
	return nil
}

func (dpkg *Dpkg) Metrics() *metrics {
	return dpkg.metrics
}
