package pacman

import (
	"a.yandex-team.ru/infra/hostctl/internal/units/env/pacman/dpkgutil"
	"fmt"
	"testing"

	"a.yandex-team.ru/infra/hostctl/internal/executil"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/test/assertpb"
	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
)

func newTestDpkg(cmds *DpkgCmds) *Dpkg {
	l, err := zap.New(zap.TSKVConfig(log.DebugLevel))
	if err != nil {
		panic(err)
	}
	return &Dpkg{
		metrics: &metrics{
			installedOk:     unistat.NewNumeric("hostctl_installed-ok", 10, aggr.Counter(), unistat.Sum),
			installFailures: unistat.NewNumeric("hostctl_installed-failures", 10, aggr.Counter(), unistat.Sum),
			fixAttempts:     unistat.NewNumeric("hostctl_fix-attempts", 10, aggr.Counter(), unistat.Sum),
			fixFailures:     unistat.NewNumeric("hostctl_fix-failures", 10, aggr.Counter(), unistat.Sum),
			purgedOk:        unistat.NewNumeric("hostctl_purged-ok", 10, aggr.Counter(), unistat.Sum),
			purgeFailures:   unistat.NewNumeric("hostctl_purged-failures", 10, aggr.Counter(), unistat.Sum),
		},
		l:    l,
		cmds: cmds,
	}
}

func TestDpkg_Repair(t *testing.T) {
	tests := []struct {
		name    string
		cmds    *DpkgCmds
		want    bool
		wantErr bool
	}{
		{
			name: "repair ok",
			cmds: &DpkgCmds{
				DpkgFix: executil.NewCmd(dpkgFix[0], dpkgFix[1:]...).WithExecutor(executil.FakeOkDpkgRepairQueryExecutor),
				AptFix:  executil.NewCmd(aptFix[0], aptFix[1:]...).WithExecutor(executil.FakeOkAptRepairQueryExecutor),
			},
			want:    true,
			wantErr: false,
		},
		{
			name: "repair fail",
			cmds: &DpkgCmds{
				DpkgFix: executil.NewCmd(dpkgFix[0], dpkgFix[1:]...).WithExecutor(executil.FakeFailDpkgRepairQueryExecutor),
				AptFix:  executil.NewCmd(aptFix[0], aptFix[1:]...).WithExecutor(executil.FakeFailAptRepairQueryExecutor),
			},
			want:    false,
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			dpkg := newTestDpkg(tt.cmds)
			got, err := dpkg.Repair()
			if (err != nil) != tt.wantErr {
				t.Errorf("Repair() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != tt.want {
				t.Errorf("Repair() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestDpkg_Install(t *testing.T) {
	install := executil.NewCmd(installPrefix[0], installPrefix[1:]...)
	type args struct {
		name    string
		version string
	}
	tests := []struct {
		name    string
		cmds    *DpkgCmds
		args    args
		want    *dpkgutil.PackageStatus
		wantErr bool
	}{{
		name: "installed pkg",
		cmds: &DpkgCmds{
			InstallPrefix: install.WithExecutor(executil.FakeOkAptInstallQueryExecutor),
		},
		args: args{
			name:    "apt",
			version: "1.2.32",
		},
		want: &dpkgutil.PackageStatus{
			Name:      "apt",
			Version:   "1.2.32",
			Installed: true,
		},
		wantErr: false,
	}, {
		name: "installed unknown pkg",
		cmds: &DpkgCmds{
			InstallPrefix: install.WithExecutor(executil.FakeFailAptInstallQueryExecutor),
		},
		args: args{
			name:    "apt",
			version: "1.2.33",
		},
		want:    nil,
		wantErr: true,
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			dpkg := newTestDpkg(tt.cmds)
			got, err := dpkg.Install(tt.args.name, tt.args.version)
			if (err != nil) != tt.wantErr {
				t.Errorf("Install() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			assertpb.Equal(t, tt.want, got, fmt.Sprintf("Install() got = %v, want %v", got, tt.want))
		})
	}
}

func TestDpkg_InstallSet(t *testing.T) {
	install := executil.NewCmd(installPrefix[0], installPrefix[1:]...)
	type args struct {
		packageSet []Package
	}
	tests := []struct {
		name    string
		cmds    *DpkgCmds
		args    args
		wantErr bool
	}{{
		name: "installed pkg",
		cmds: &DpkgCmds{
			InstallPrefix: install.WithExecutor(executil.FakeOkAptInstallQueryExecutor),
		},
		args: args{
			packageSet: []Package{{
				Name:    "apt",
				Version: "1.2.32",
			}},
		},
		wantErr: false,
	}, {
		name: "installed two pkg",
		cmds: &DpkgCmds{
			InstallPrefix: install.WithExecutor(executil.FakeOkAptInstallTwoPkgQueryExecutor),
		},
		args: args{
			packageSet: []Package{{
				Name:    "apt",
				Version: "1.2.32",
			}, {
				Name:    "git",
				Version: "1:2.7.4-0ubuntu1.9",
			}},
		},
		wantErr: false,
	}, {
		name: "installed unknown pkg",
		cmds: &DpkgCmds{
			InstallPrefix: install.WithExecutor(executil.FakeFailAptInstallQueryExecutor),
		},
		args: args{
			packageSet: []Package{{
				Name:    "apt",
				Version: "1.2.33",
			}},
		},
		wantErr: true,
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			dpkg := newTestDpkg(tt.cmds)
			if err := dpkg.InstallSet(tt.args.packageSet); (err != nil) != tt.wantErr {
				t.Errorf("InstallSet() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}

func TestDpkg_Purge(t *testing.T) {
	purge := executil.NewCmd(purgePrefix[0], purgePrefix[1:]...)
	type args struct {
		pName string
	}
	tests := []struct {
		name    string
		cmds    *DpkgCmds
		args    args
		wantErr bool
	}{{
		name: "purge ok",
		cmds: &DpkgCmds{
			PurgePrefix: purge.WithExecutor(executil.FakeOkAptPurgeQueryExecutor),
		},
		args: args{
			pName: "git",
		},
		wantErr: false,
	}, {
		name: "purge fail",
		cmds: &DpkgCmds{
			PurgePrefix: purge.WithExecutor(executil.FakeFailAptPurgeQueryExecutor),
		},
		args: args{
			pName: "apt",
		},
		wantErr: true,
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			dpkg := newTestDpkg(tt.cmds)
			if err := dpkg.Purge(tt.args.pName); (err != nil) != tt.wantErr {
				t.Errorf("Purge() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}

func TestDpkg_PurgeSet(t *testing.T) {
	purge := executil.NewCmd(purgePrefix[0], purgePrefix[1:]...)
	type args struct {
		names []string
	}
	tests := []struct {
		name    string
		cmds    *DpkgCmds
		args    args
		wantErr bool
	}{{
		name: "purge ok",
		cmds: &DpkgCmds{
			PurgePrefix: purge.WithExecutor(executil.FakeOkAptPurgeQueryExecutor),
		},
		args: args{
			names: []string{"git"},
		},
		wantErr: false,
	}, {
		name: "purge fail",
		cmds: &DpkgCmds{
			PurgePrefix: purge.WithExecutor(executil.FakeFailAptPurgeQueryExecutor),
		},
		args: args{
			names: []string{"apt"},
		},
		wantErr: true,
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			dpkg := newTestDpkg(tt.cmds)
			if err := dpkg.PurgeSet(tt.args.names); (err != nil) != tt.wantErr {
				t.Errorf("PurgeSet() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}
