package dpkgutil

import (
	"a.yandex-team.ru/library/go/core/resource"
	"bytes"
	"fmt"
	"github.com/stretchr/testify/assert"
	"strings"
	"testing"
)

type testPkg struct {
	installed bool
	name      string
	version   string
}

var (
	invalidPackageSeparator = `Package:python-apt-common
Status: install ok installed
Priority: optional
Section: python
Installed-Size: 232
Maintainer: Ubuntu Developers <ubuntu-devel-discuss@lists.ubuntu.com>
Architecture: all
Source: python-apt
Version: 1.1.0~beta1ubuntu0.16.04.8
Replaces: python-apt (<< 0.7.98+nmu1)
Depends: python | python3
Breaks: python-apt (<< 0.7.98+nmu1)
Enhances: python-apt, python3-apt
Description: Python interface to libapt-pkg (locales)
 The apt_pkg Python interface will provide full access to the internal
 libapt-pkg structures allowing Python programs to easily perform a
 variety of functions.
 .
 This package contains locales.
Original-Maintainer: APT Development Team <deity@lists.debian.org>`
	invalidPackageEmptyStatusAndVersion = `Package: python-apt-common`
)

func Test(t *testing.T) {
	assertedPkgs := []*testPkg{
		{
			installed: true,
			name:      "apt",
			version:   "1.2.29ubuntu0.1",
		}, {
			installed: true,
			name:      "bash",
			version:   "4.3-14ubuntu1.2",
		}, {
			installed: false,
			name:      "xz_chto",
			version:   "unknown",
		},
	}
	status := resource.Get("/var/lib/dpkg/status")
	fakeFile := bytes.Buffer{}
	fakeFile.Write(status)
	pkgs := map[string]*PackageStatus{
		"apt": {
			Installed: false,
			Name:      "apt",
			Version:   "unknown",
		},
		"bash": {
			Installed: false,
			Name:      "bash",
			Version:   "unknown",
		},
		"xz_chto": {
			Installed: false,
			Name:      "xz_chto",
			Version:   "unknown",
		},
	}
	err := readStatuses(&fakeFile, pkgs)
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, len(pkgs), 3)
	for _, pkg := range assertedPkgs {
		p, found := pkgs[pkg.name]
		assert.Equal(t, true, found, fmt.Sprintf("%s not found in /var/lib/dpkg/status", pkg.name))
		if !found {
			continue
		}
		assert.Equal(t, pkg.name, p.Name, fmt.Sprintf("%s name in /var/lib/dpkg/status is %s", pkg.name, p.Name))
		assert.Equal(t, pkg.version, p.Version, fmt.Sprintf("%s version in /var/lib/dpkg/status is %s", pkg.name, p.Version))
		assert.Equal(t, pkg.installed, p.Installed, fmt.Sprintf("%s installed in /var/lib/dpkg/status is %t", pkg.name, p.Installed))
	}
}

func TestInvalidSeparator(t *testing.T) {
	b := bytes.NewBuffer([]byte(invalidPackageSeparator))
	s := NewPackageScanner(b)
	for s.Scan() {
		_, err := s.Package()
		assert.Errorf(t, err, fmt.Sprintf("failed to unmarshal pkg one of [%s] was not unmarshaled", strings.Join(knownKeys, ",")))
	}
}

func TestInvalidEmptyVersionAndStatus(t *testing.T) {
	b := bytes.NewBuffer([]byte(invalidPackageEmptyStatusAndVersion))
	s := NewPackageScanner(b)
	for s.Scan() {
		_, err := s.Package()
		assert.Errorf(t, err, fmt.Sprintf("failed to unmarshal pkg one of [%s] was not unmarshaled", strings.Join(knownKeys, ",")))
	}
}

func BenchmarkPackageScanner(b *testing.B) {
	b.ReportAllocs()
	status := resource.Get("/var/lib/dpkg/status")
	fakeFile := bytes.Buffer{}
	fakeFile.Write(status)
	pkgs := map[string]*PackageStatus{
		"apt": {
			Installed: false,
			Name:      "apt",
			Version:   "unknown",
		},
		"bash": {
			Installed: false,
			Name:      "bash",
			Version:   "unknown",
		},
		"xz_chto": {
			Installed: false,
			Name:      "xz_chto",
			Version:   "unknown",
		},
	}
	for i := 0; i < b.N; i++ {
		err := readStatuses(&fakeFile, pkgs)
		if err != nil {
			b.Error(err)
		}
	}
}
