package dpkgutil

import (
	"bufio"
	"bytes"
	"errors"
	"fmt"
	"io"
	"strings"
)

type PackageStatus struct {
	Installed bool
	Name      string
	Version   string
}

func (s *PackageStatus) String() string {
	return fmt.Sprintf("(name=%s,version=%s,installed=%t)", s.Name, s.Version, s.Installed)
}

func (s *PackageStatus) CopyFrom(p *PackageStatus) {
	s.Version = p.Version
	s.Name = p.Name
	s.Installed = p.Installed
}

type PackageScanner struct {
	s               *bufio.Scanner
	p               *PackageStatus
	pkgFieldsBuffer map[string]string
}

func NewPackageScanner(f io.Reader) *PackageScanner {
	s := bufio.NewScanner(f)
	s.Split(scanPkg)
	return &PackageScanner{
		s:               s,
		p:               &PackageStatus{},
		pkgFieldsBuffer: make(map[string]string),
	}
}

func (s *PackageScanner) Scan() bool {
	return s.s.Scan()
}

func (s *PackageScanner) Package() (*PackageStatus, error) {
	Unmarshal(s.s.Bytes(), s.pkgFieldsBuffer)
	visited := make(map[string]bool)
	for k, v := range s.pkgFieldsBuffer {
		switch k {
		case "Status":
			installed, err := isStatusInstalled(v)
			if err != nil {
				return nil, err
			}
			s.p.Installed = installed
			visited[k] = true
		case "Package":
			s.p.Name = v
			visited[k] = true
		case "Version":
			s.p.Version = v
			visited[k] = true
		}
	}
	// we need to unmarshal all this fields
	if !visited["Status"] || !visited["Package"] || !visited["Version"] {
		return nil, fmt.Errorf("failed to unmarshal pkg one of [%s] was not unmarshaled", strings.Join(knownKeys, ","))
	}
	return s.p, nil
}

var (
	wantInfos = map[string]int{
		"unknown":   WantUnknown,
		"install":   WantInstall,
		"hold":      WantHold,
		"deinstall": WantDeinstall,
		"purge":     WantPurge,
	}
	flagInfos = map[string]int{
		"ok":        FlagOK,
		"reinstreq": FlagReinstallRequired,
	}
	statusInfos = map[string]int{
		"not-installed":    StatusNotInstalled,
		"config-files":     StatusConfigFiles,
		"half-installed":   StatusHalfInstalled,
		"unpacked":         StatusUnpacked,
		"half-configured":  StatusHalfConfigured,
		"triggers-awaited": StatusTriggersAwaited,
		"triggers-pending": StatusTriggersPending,
		"installed":        StatusInstalled,
	}
)

func isStatusInstalled(s string) (bool, error) {
	part := strings.SplitN(s, " ", 3)
	if len(part) != 3 {
		return false, errors.New("dpkg: error in status field")
	}
	want := part[0]
	flag := part[1]
	status := part[2]
	if _, ok := wantInfos[want]; !ok {
		return false, fmt.Errorf("dpkg: invalid want %s in status", want)
	}
	if _, ok := flagInfos[flag]; !ok {
		return false, fmt.Errorf("dpkg: invalid flag %s in status", flag)
	}
	if _, ok := statusInfos[status]; !ok {
		return false, fmt.Errorf("dpkg: invalid status %s", status)
	}
	return statusInfos[status] == StatusInstalled, nil
}

// patched bufio.ScanLine for pkgs
// scans blocks by '\n\n'
func scanPkg(data []byte, atEOF bool) (advance int, token []byte, err error) {
	if atEOF && len(data) == 0 {
		return 0, nil, nil
	}
	if i := bytes.Index(data, []byte("\n\n")); i >= 0 {
		return i + 2, data[0:i], nil
	}
	if atEOF {
		return len(data), data, nil
	}
	return 0, nil, nil
}
