package pacman

import (
	"a.yandex-team.ru/infra/hostctl/internal/units/env/pacman/dpkgutil"
	"bytes"
	"errors"
	"fmt"
	"syscall"
	"time"

	"a.yandex-team.ru/library/go/yandex/unistat"
)

type Metrics interface {
	Metrics()
}

type metrics struct {
	installedOk     *unistat.Numeric
	installFailures *unistat.Numeric
	fixAttempts     *unistat.Numeric
	fixFailures     *unistat.Numeric
	purgedOk        *unistat.Numeric
	purgeFailures   *unistat.Numeric
}

type Package struct {
	Name    string
	Version string
}

func (p *Package) String() string {
	return fmt.Sprintf("%s=%s", p.Name, p.Version)
}

type PackageManager interface {
	Update() error
	GetPackageStatus(name string) (*dpkgutil.PackageStatus, error)
	List(packages []string) (map[string]*dpkgutil.PackageStatus, error)
	Install(name, version string) (*dpkgutil.PackageStatus, error)
	InstallDryRun(name, version string) (*dpkgutil.PackageStatus, error)
	InstallSet(packageSet []Package) error
	InstallSetDryRun(packageSet []Package) error
	Purge(pName string) error
	PurgeSet(names []string) error
	Repair() (bool, error)
}

var (
	DpkgEnvVars = []string{
		"APT_LISTBUGS_FRONTEND=none",
		"APT_LISTCHANGES_FRONTEND=none",
		"DEBIAN_FRONTEND=noninteractive",
		"UCF_FORCE_CONFFOLD=1",
		// We parse output, ensure locale
		"LANG=en_US.UTF-8",
		// Put this env (this is documented API, do not remove)
		"HOSTMAN=1",
	}
	// We use really long timeouts because on pressurised hosts
	// simple actions can take a long amount of time.
	DpkgFixTimeout = 10 * time.Minute
	AptGetTimeout  = 10 * time.Minute // apt-get update run timeout
	InstallTimeout = 10 * time.Minute
	PurgeTimeout   = 10 * time.Minute
	SetTimeout     = InstallTimeout * 3
)

func getTmpDir(envtmp string) (string, error) {
	err := errors.New("no suitable temp dirs")
	ok := false
	if envtmp != "" {
		ok, err = hasFreeSpace(envtmp)
	}
	if err == nil && ok {
		return envtmp, nil
	}
	// Let's try /tmp too
	if envtmp != "/tmp" {
		ok, err = hasFreeSpace("/tmp")
	}
	if err == nil && ok {
		return "/tmp", nil
	}
	return "", err
}

func hasFreeSpace(path string) (bool, error) {
	stat := &syscall.Statfs_t{}
	err := syscall.Statfs(path, stat)
	if err != nil {
		return false, err
	}
	return stat.Bavail > 0, nil
}

// Checks if there is something wrong that can be fixed.
func dpkgNeedsFix(stderr []byte) bool {
	return bytes.Contains(stderr, []byte("dpkg was interrupted, you must manually run"))
}
