package porto

import (
	pb "a.yandex-team.ru/infra/hostctl/proto"
	porto "a.yandex-team.ru/infra/porto/api_go"
	portopb "a.yandex-team.ru/infra/porto/proto"
	"fmt"
	"strings"
	"syscall"
	"time"
)

// Really big timeout on overloaded hosts.
const productionTimeout = 1 * time.Minute

func NewProduction() *Real {
	c, _ := porto.Dial()
	c.SetTimeout(productionTimeout)
	return &Real{c}
}

type Status struct {
	Exist   bool
	Running bool
	Etag    string
}

type Real struct {
	conn porto.PortoAPI
}

func (d *Real) Start(name, etag string, props *pb.PortoProperties) error {
	optionals := map[string]string{
		"virt_mode":        props.VirtMode,
		"isolate":          props.Isolate,
		"user":             props.User,
		"group":            props.Group,
		"memory_guarantee": props.MemoryGuarantee,
		"memory_limit":     props.MemoryLimit,
		"cpu_guarantee":    props.CpuGuarantee,
		"cpu_limit":        props.CpuLimit,
		"respawn":          "true",
		"respawn_delay":    fmt.Sprintf("%d", props.RespawnDelay),
		"max_respawns":     fmt.Sprintf("%d", props.MaxRespawns),
	}
	if err := d.conn.Create(name); err != nil {
		return fmt.Errorf("failed to create: %w", err)
	}
	if err := d.conn.SetProperty(name, "command", strings.Join(props.Cmd, " ")); err != nil {
		return fmt.Errorf("failed to set command: %w", err)
	}
	// Set optional properties
	for k, v := range optionals {
		// Skip empty options, because porto e.g does not accept cpu_guarantee as ""
		if v == "" {
			continue
		}
		if err := d.conn.SetProperty(name, k, v); err != nil {
			return fmt.Errorf("failed to set %s=%s: %w", k, v, err)
		}
	}
	// Set controllers (their API is a bit different)
	for k, v := range props.Controllers {
		if err := d.conn.SetProperty(name, fmt.Sprintf("controllers[%s]", k), v); err != nil {
			return fmt.Errorf("failed to set controllers[%s]=%s: %w", k, v, err)
		}
	}
	// Set ulimits
	for k, v := range props.Ulimit {
		if err := d.conn.SetProperty(name, fmt.Sprintf("ulimit[%s]", k), v); err != nil {
			return fmt.Errorf("failed to set ulimit[%s]=%s: %v", k, v, err)
		}
	}
	// Set capabilities - we have list of strings,
	// porto wants `;` separated
	for k, v := range map[string][]string{
		"capabilities":         props.Capabilities,
		"capabilities_ambient": props.CapabilitiesAmbient,
	} {
		if len(v) == 0 {
			continue
		}
		if err := d.conn.SetProperty(name, k, strings.Join(v, ";")); err != nil {
			return fmt.Errorf("failed to set %s=%s: %w", k, v, err)
		}
	}
	if err := d.conn.SetProperty(name, "private", etagToPrivate(etag)); err != nil {
		return fmt.Errorf("failed to set private: %w", err)
	}
	if err := d.conn.Start(name); err != nil {
		return fmt.Errorf("failed to start: %w", err)
	}
	return nil
}

func (d *Real) Kill(name string, sig syscall.Signal) error {
	err := d.conn.Kill(name, sig)
	if err == nil {
		return nil
	}
	e, ok := err.(*porto.PortoError)
	if !ok {
		return err
	}
	if e.Code == portopb.EError_InvalidState {
		// Must be "dead" container, which is okay
		return nil
	}
	if e.Code == portopb.EError_ContainerDoesNotExist {
		return nil
	}
	return err
}

func (d *Real) Destroy(name string) error {
	err := d.conn.Destroy(name)
	if err == nil {
		return nil
	}
	e, ok := err.(*porto.PortoError)
	if !ok {
		return err
	}
	if e.Code == portopb.EError_InvalidState {
		// Must be "dead" container, which is okay
		return nil
	}
	if e.Code == portopb.EError_ContainerDoesNotExist {
		return nil
	}
	return err
}

func (d *Real) Get(name, prop string) (string, string, error) {
	val, err := d.conn.GetProperty(name, prop)
	if err != nil {
		e, ok := err.(*porto.PortoError)
		if !ok {
			return "", "", err
		}
		if e.Code == portopb.EError_InvalidState {
			// Must be "dead" container, which is okay
			return "", "", nil
		}
		if e.Code == portopb.EError_ContainerDoesNotExist {
			return "", "", nil
		}
		return "", "", err
	}
	private, err := d.conn.GetProperty(name, "private")
	if err != nil {
		return "", "", err
	}
	if !strings.HasPrefix(private, CookiePrefix) {
		return "", val, nil
	}
	return etagFromPrivate(private), val, nil
}

func (d *Real) IsRunning(name string) (string, bool, error) {
	etag, val, err := d.Get(name, "state")
	return etag, val == "running", err
}

func (d *Real) Status(name string) (*Status, error) {
	state, err := d.conn.GetProperty(name, "state")
	if err != nil {
		e, ok := err.(*porto.PortoError)
		if !ok {
			return nil, err
		}
		if e.Code == portopb.EError_InvalidState {
			// Must be "dead" container, which is okay
			return &Status{Exist: true, Running: false, Etag: ""}, nil
		}
		if e.Code == portopb.EError_ContainerDoesNotExist {
			return &Status{Exist: false, Running: false, Etag: ""}, nil
		}
		return nil, err
	}
	private, err := d.conn.GetProperty(name, "private")
	if err != nil {
		return nil, err
	}
	if !strings.HasPrefix(private, CookiePrefix) {
		return &Status{Exist: true, Running: state == "running"}, nil
	}
	etag := etagFromPrivate(private)
	return &Status{Exist: true, Running: state == "running", Etag: etag}, nil
}

func (d *Real) Disable(name string) error {
	err := d.conn.SetProperty(name, "respawn", "false")
	if err == nil {
		return nil
	}
	e, ok := err.(*porto.PortoError)
	if !ok {
		return err
	}
	if e.Code == portopb.EError_InvalidState {
		// Must be "dead" container, which is okay
		return nil
	}
	if e.Code == portopb.EError_ContainerDoesNotExist {
		return nil
	}
	return err
}

func (d *Real) WaitDead(name string, timeout time.Duration, timeFun func() time.Time) (bool, error) {
	// Porto has .Wait API but it can return early
	// if porto restart, so too be on the safe side - just poll
	deadline := timeFun().Add(timeout)
	for {
		_, running, err := d.IsRunning(name)
		if err == nil && !running {
			return true, nil
		}
		if timeFun().After(deadline) {
			return false, err
		}
		time.Sleep(time.Millisecond * 250)
	}
}
