package units

import (
	"fmt"
	"path"
	"strings"
	"time"

	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/template"
	"a.yandex-team.ru/infra/hostctl/internal/unit/kind"
	"a.yandex-team.ru/infra/hostctl/pkg/hostinfo"
	"a.yandex-team.ru/infra/hostctl/pkg/unitstorage"

	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/slices"
	"github.com/golang/protobuf/ptypes"

	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/manager"
	"a.yandex-team.ru/infra/hostctl/internal/units/report"
	"a.yandex-team.ru/infra/hostctl/internal/units/revjob"
	"a.yandex-team.ru/infra/hostctl/internal/units/storage"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks"
)

func NewManageOpts(e *env.Env, h *pb.HostInfo, s storage.Storage, overrideDirs []string, repoDir string) *manageOpts {
	return &manageOpts{
		e:            e,
		h:            h,
		s:            s,
		overrideDirs: overrideDirs,
		repoDir:      repoDir,
	}
}

func NewRemoveOpts(e *env.Env, h *pb.HostInfo, s storage.Storage, name string) *removeOpts {
	return &removeOpts{
		e:    e,
		h:    h,
		s:    s,
		name: name,
	}
}

type manageOpts struct {
	e            *env.Env
	h            *pb.HostInfo
	s            storage.Storage
	overrideDirs []string
	repoDir      string
}

type removeOpts struct {
	e    *env.Env
	h    *pb.HostInfo
	s    storage.Storage
	name string
}

func ManageAll(opts *manageOpts) error {
	s, err := unitstorage.NewDefaultFSStorage(unitstorage.NewLocalFS(), opts.repoDir, opts.overrideDirs)
	if err != nil {
		return err
	}
	all, err := s.DiscoverUnits()
	if err != nil {
		return err
	}
	incoming := make(map[string]template.Template)
	for _, u := range all {
		t, err := template.FromStorage(s, u)
		if err != nil {
			return err
		}
		incoming[u] = t
	}
	for _, t := range incoming {
		opts.e.L.Infof("%s found in %s origin", t.Name(), t.Path())
	}
	//goland:noinspection GoNilness
	targetSlots := execute(opts.e, opts.h, incoming, opts.s, manager.NewTarget())
	return checkStatus(targetSlots)
}

func ManageTargets(opts *manageOpts, tu *manager.Target) error {
	s, err := unitstorage.NewDefaultFSStorage(unitstorage.NewLocalFS(), opts.repoDir, opts.overrideDirs)
	if err != nil {
		return err
	}
	incoming := make(map[string]template.Template)
	for _, name := range tu.UnitsNames() {
		t, err := template.FromStorage(s, name)
		if err != nil {
			return err
		}
		incoming[name] = t
	}
	err = checkAllTargetUnitsPresent(incoming, tu)
	if err != nil {
		return err
	}
	//goland:noinspection GoNilness
	targetSlots := execute(opts.e, opts.h, incoming, opts.s, tu)
	return checkStatus(targetSlots)
}

func ManageInline(opts *manageOpts, inlineYaml string) error {
	t, err := template.FromReader(strings.NewReader(inlineYaml), "<stdin>", "<stdin>")
	if err != nil {
		return err
	}
	tu := manager.NewStrictTarget([]string{t.Name()})
	//goland:noinspection GoNilness
	targetSlots := execute(opts.e, opts.h, map[string]template.Template{t.Name(): t}, opts.s, tu)
	return checkStatus(targetSlots)
}

func Remove(opts *removeOpts) error {
	incoming := template.Removed(opts.name)
	tu := manager.NewStrictTarget([]string{incoming.Name()})
	//goland:noinspection GoNilness
	targetSlots := execute(opts.e, opts.h, map[string]template.Template{incoming.Name(): incoming}, opts.s, tu)
	return checkStatus(targetSlots)
}

func checkAllTargetUnitsPresent(units map[string]template.Template, tu *manager.Target) error {
	missing := checkMissingUnits(units, tu)
	if len(missing) == 0 {
		return nil
	}
	return fmt.Errorf("target units missing: [%s]", strings.Join(missing, ", "))
}

func checkStatus(slots map[string]slot.Slot) error {
	errs := make([]string, 0)
	for name, s := range slots {
		if !s.Status().Ok() {
			if s.Status().IsPending() {
				errs = append(errs, fmt.Sprintf("%s pending: %s", name, s.Status().Pending.Message))
			}
			if s.Status().IsRemoved() {
				continue
			}
			if !s.Status().IsReady() {
				errs = append(errs, fmt.Sprintf("%s not ready: %s", name, s.Status().Ready.Message))
			}
		}
	}
	if len(errs) == 0 {
		return nil
	}
	return fmt.Errorf("some units not ready: [%s]", strings.Join(errs, ", "))
}

func ReportState(l log.Logger, hi *pb.HostInfo, hm report.Reporter, juggler report.Reporter, yasmReporterBuilder report.YasmReporterBuilder, ctype string) error {
	l.Info("Reporting...")
	s := storage.New("")
	l.Info("Loading state from file system...")
	state, err := s.Load()
	if err != nil {
		l.Errorf("Failed to read hostctl state: %s", err.Error())
	} else {
		unitsTS, err := ptypes.Timestamp(state.UnitsTs)
		if err != nil {
			l.Errorf("Failed to read units_ts from state: %s", err.Error())
		}
		l.Debugf("units_ts recovered from state: %s", unitsTS)
	}
	if err != nil {
		l.Errorf("Failed to read hostctl state: %s", err.Error())
		return err
	}
	rebooted := handleReboot(l, state)
	if err := s.SaveIfModified(l, state); err != nil {
		l.Errorf("Failed to save state: %s", err.Error())
	}
	yasm := yasmReporterBuilder(map[string]string{
		"ctype": ctype,
		"prj":   hi.WalleProject,
	}, l, &report.Config{Hostname: hi.Hostname}, rebooted)
	unitsTS, err := ptypes.Timestamp(state.UnitsTs)
	if err != nil {
		return err
	}
	l.Infof("Using units-ts=%s", unitsTS)
	return Report(slot.SlotsFromPb(state), hi, []report.Reporter{yasm, hm, juggler}, l, unitsTS)
}

/*
handleReboot:
* Read current boot id
* Compare with current
* Update state with current
*/
func handleReboot(l log.Logger, state *pb.HostctlState) bool {
	curBootID, err := hostinfo.BootID()
	if err != nil {
		// Do not modify state and work in regular mode
		l.Errorf("Failed to get boot_id: %s", err)
		return false
	}
	stateBootID := ""
	if state.Monitoring != nil {
		stateBootID = state.Monitoring.BootId
	}
	rebooted := isRebooted(stateBootID, curBootID)
	l.Infof("Current boot_id: '%s', from state: '%s'", curBootID, stateBootID)
	if rebooted {
		l.Info("Boot id changed, host has rebooted")
	}
	updateBootID(state, curBootID)
	return rebooted
}

// isRebooted compare current boot id with state
// to handle first run after reboot.
func isRebooted(fromState, current string) bool {
	// Hostctl initialization
	if fromState == "" {
		// When we lost state, safer to work in regular mode
		return false
	}
	return current != fromState
}

func updateBootID(state *pb.HostctlState, bootID string) {
	if state.Monitoring == nil {
		state.Monitoring = &pb.HostctlState_Monitoring{}
	}
	state.Monitoring.BootId = bootID
}

func execute(e *env.Env, info *pb.HostInfo, incoming map[string]template.Template, s storage.Storage, tu *manager.Target) map[string]slot.Slot {
	state, err := s.Load()
	if err != nil {
		e.L.Errorf("Failed to read hostctl state: %s", err.Error())
	}
	ctl := manager.NewHostCtl(info, state)
	e.L.Info("Applying incoming units to tx...")
	// filtering units by target
	incomingToApply := tu.TargetUnits(incoming)
	ctl.ApplyMany(incomingToApply, e)
	// Save our state (if changed) to enforce invariant:
	// All actions performed are derived from persisted units.
	// Thus we can rollback any side effects (packages, services).
	e.L.Info("Saving state to fs...")
	unitsTSProto, err := ptypes.TimestampProto(ctl.UnitsTS())
	if err != nil {
		// timestamp always ok
		panic(err)
	}
	state.Slots = slot.HostCTLSlots(ctl.Slots)
	state.UnitsTs = unitsTSProto
	if err := s.SaveIfModified(e.L, state); err != nil {
		e.L.Errorf("Failed to save state: %s", err.Error())
	}
	e.L.Info("Executing units...")
	targetSlots := tu.TargetSlots(ctl.Slots)
	ctl.Execute(targetSlots, e)
	// getting units_ts before gc units. to save ts of units removing
	unitsTS := ctl.UnitsTS()
	e.L.Info("Garbage collecting old revs...")
	ctl.GCSuccessfullyRemovedRevs(e.L)
	e.L.Info("Garbage collecting removed units...")
	ctl.GCRemovedSlots(e.L)
	e.L.Info("Saving state to fs...")
	unitsTSProto, err = ptypes.TimestampProto(unitsTS)
	if err != nil {
		// timestamp always ok
		panic(err)
	}
	state.Slots = slot.HostCTLSlots(ctl.Slots)
	state.UnitsTs = unitsTSProto
	if err := s.SaveIfModified(e.L, state); err != nil {
		e.L.Errorf("Failed to save state: %s", err.Error())
	}
	return targetSlots
}

func checkMissingSlots(slots map[string]slot.Slot, units []template.Template) []string {
	missingSlots := make([]string, 0)
	for slotName := range slots {
		missing := true
		for _, u := range units {
			if u.Name() == slotName {
				missing = false
			}
		}
		if missing {
			missingSlots = append(missingSlots, slotName)
		}
	}
	return missingSlots
}

func checkMissingUnits(units map[string]template.Template, t *manager.Target) []string {
	namesToCheck := make([]string, 0, len(units))
	for _, u := range units {
		namesToCheck = append(namesToCheck, u.Name())
	}
	missing := make([]string, 0, len(units))
	for _, name := range t.UnitsNames() {
		if !slices.ContainsString(namesToCheck, name) {
			missing = append(missing, name)
		}
	}
	return missing
}

// Apply unit and do not save hostctl state to fs after execution
func Apply(e *env.Env, hi *pb.HostInfo, unitPath string) (*slot.Status, error) {
	e.L.Info("Loading hostctl state from file system...")
	state, err := storage.NewReadonly("").Load()
	if err != nil {
		e.L.Errorf("Failed to read hostctl state: %s", err.Error())
		state = &pb.HostctlState{}
	}
	ctl := manager.NewHostCtl(hi, state)
	e.L.Infof("Reading unit from '%s'...", unitPath)
	s, err := unitstorage.NewDefaultFSStorage(unitstorage.NewLocalFS(), "", []string{path.Dir(unitPath)})
	if err != nil {
		return nil, err
	}
	unitFile := path.Base(unitPath)
	t, err := template.FromStorage(s, unitFile[:len(unitFile)-5])
	if err != nil {
		return nil, err
	}
	changed, err := ctl.Apply(t, e)
	if err != nil {
		return nil, err
	}
	if !changed {
		if s := ctl.Slots[t.Name()]; s != nil {
			return s.Status(), nil
		}
		// may be we had stage == 'absent' or delete_requested
		return nil, fmt.Errorf("nothing was changed")
	}
	sl := ctl.Slots[t.Name()]
	if s == nil {
		return nil, fmt.Errorf("slot is absent, nothing to do")
	}
	ctl.Execute(map[string]slot.Slot{sl.Name(): sl}, e)
	return sl.Status(), nil
}

// Get Plans for unit's jobs
func Plan(e *env.Env, hi *pb.HostInfo, unitPath string, s storage.Storage) (map[string]tasks.Plan, error) {
	e.L.Info("Loading slots from file system...")
	state, _ := s.Load()
	//goland:noinspection GoNilness
	ctl := manager.NewHostCtl(hi, state)
	e.L.Infof("Reading unit from '%s'...", unitPath)
	fss, err := unitstorage.NewDefaultFSStorage(unitstorage.NewLocalFS(), "", []string{path.Dir(unitPath)})
	if err != nil {
		return nil, err
	}
	unitFile := path.Base(unitPath)
	t, err := template.FromStorage(fss, unitFile[:len(unitFile)-5])
	if err != nil {
		return nil, err
	}
	changed, err := ctl.Apply(t, e)
	if err != nil {
		return nil, fmt.Errorf("failed to apply unit: %w", err)
	}
	// Render manually again to print resulting format,
	// without checking for errors (as Apply has just done that too)
	u, _ := t.Render(hi)
	formatted, err := u.Prettify()
	if err != nil {
		e.L.Errorf("Failed to fmt t unit: %s", err.Error())
	} else {
		e.L.Infof("\n%s", formatted)
	}
	if !changed {
		// we need to return not nil plan's because we'll marshal it later
		return make(map[string]tasks.Plan), nil
	}
	e.L.Info("Getting unit plan...")
	return ctl.ExecuteManager.Plan(ctl.Slots[t.Name()], e.L), nil
}

func Restart(e *env.Env, unitName string, st storage.Storage) ([]*changelog.ChangeLog, error) {
	chl := make([]*changelog.ChangeLog, 0)
	state, err := st.Load()
	if err != nil {
		return nil, err
	}
	s := getSlotByName(state, unitName)
	cur, err := Current(s)
	if err != nil {
		return chl, err
	}
	switch cur.Kind() {
	case kind.TimerJob:
		fallthrough
	case kind.SystemService:
		fallthrough
	case kind.PortoDaemon:
		break
	default:
		return chl, fmt.Errorf("cannot restart %s", cur.Kind())
	}
	job := revjob.NewRestartFromSlotAndRev(unitName, cur, s)
	p, err := revjob.Plan(job).Fmt(job.SlotName(), job.RevID(), job.Description())
	if err != nil {
		e.L.Errorf("Failed to prettify plan: %s", err.Error())
	} else {
		e.L.Infof("\n%s", p)
	}
	e.L.Info("Executing restart job...")
	ch, err := job.Execute(e)
	if err != nil {
		return chl, err
	}
	chl = append(chl, ch)
	return chl, nil
}

func Reinstall(e *env.Env, unitName string, st storage.Storage) ([]*changelog.ChangeLog, error) {
	chl := make([]*changelog.ChangeLog, 0)
	state, err := st.Load()
	if err != nil {
		return nil, err
	}
	s := getSlotByName(state, unitName)
	cur, err := Current(s)
	if err != nil {
		return chl, err
	}
	job := revjob.NewReinstallFromRevAndSlot(unitName, cur, s)
	if job == nil {
		return chl, fmt.Errorf("cannot reinstall %s", cur.Kind())
	}
	p, err := revjob.Plan(job).Fmt(job.SlotName(), job.RevID(), job.Description())
	if err != nil {
		e.L.Errorf("Failed to prettify plan: %s", err.Error())
	} else {
		e.L.Infof("\n%s", p)
	}
	e.L.Info("Executing reinstall job...")
	ch, err := job.Execute(e)
	if err != nil {
		return chl, err
	}
	chl = append(chl, ch)
	return chl, nil
}

// Read slots from fs and find slot with name == slotName
func Slot(slotName string, s storage.Storage) (slot.Slot, error) {
	state, err := s.Load()
	if err != nil {
		return nil, err
	}
	return getSlotByName(state, slotName), nil
}

// Read slots from fs and find slot with name == slotName and find current revision
func Current(slot slot.Slot) (*slot.Rev, error) {
	if slot == nil {
		return nil, fmt.Errorf("unit was not found")
	}
	cur := slot.Current()
	if cur == nil {
		return nil, fmt.Errorf("current revision was not found")
	}
	return cur, nil
}

func Report(slots map[string]slot.Slot, hostInclude *pb.HostInfo, reporters []report.Reporter, l log.Logger, unitsTS time.Time) error {
	errs := make([]string, 0)
	for _, rep := range reporters {
		l.Infof("Sending report with %s...", rep.Description())
		err := rep.Report(slots, hostInclude, unitsTS)
		if err != nil {
			l.Warnf("Failed to send report with %s: %v", rep.Description(), err)
			errs = append(errs, err.Error())
		}
	}
	return fmt.Errorf("failed to report state: %s", strings.Join(errs, "; "))
}

func getSlotByName(slots *pb.HostctlState, name string) slot.Slot {
	for _, s := range slots.Slots {
		if s.Name == name {
			return slot.NewSlot(s)
		}
	}
	return nil
}
