package manager

import (
	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/unit"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/revjob"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks"
	pb "a.yandex-team.ru/infra/hostctl/proto"
)

/*
CheckConflict validates that pushing incoming revision will not be conflicted

We are trying to find conflicts between all slots and with one incoming unit.
How it works:
* Collecting revision jobs from all slots in one transaction
* Creating and pushing revision from incoming unit to transaction
* Checking conflicts between specs (no packages with different versions, no files with different content, etc)
* Trying to 'apt install --dry-run' all packages from all units in one transaction
*/
func CheckConflict(e *env.Env, slots map[string]slot.Slot, incoming *unit.Unit) error {
	txRevs := collectTransactionRevs(slots, incoming)
	txJobs := collectTransactionJobs(txRevs)
	return checkConflicts(e, txJobs)
}

func collectTransactionJobs(revs map[string][]*slot.Rev) []*revjob.RevisionJob {
	txJobs := make([]*revjob.RevisionJob, 0)
	for name, slotRevs := range revs {
		for _, rev := range slotRevs {
			txJobs = append(txJobs, revjob.FromRev(rev, slot.NewEmptySlotStatus(), name))
		}
	}
	return txJobs
}

func collectTransactionRevs(slots map[string]slot.Slot, incoming *unit.Unit) map[string][]*slot.Rev {
	txRevs := make(map[string][]*slot.Rev)
	changed := slots[incoming.Name()]
	for name, s := range slots {
		txRevs[name] = s.Revs()
	}
	// Fast path if unit was not changed
	if changed == nil || changed.Current() != nil && incoming.ID() == changed.Current().ID() {
		return txRevs
	}
	// Keep all removed revs
	slotRevs := make([]*slot.Rev, len(changed.Revs()))
	for i, rev := range changed.Removed() {
		slotRevs[i] = rev
	}
	// Switch current -> removed & incoming -> current
	// !Do not mutate real revision from state!
	if c := changed.Current(); c != nil {
		slotRevs[len(changed.Removed())] = removedFromCurrent(c)
	}
	// incoming -> current (only if not absent)
	if !incoming.Absent() {
		slotRevs = append(slotRevs, slot.RevisionFromUnit(incoming, &pb.RevisionMeta{}))
	}
	txRevs[changed.Name()] = slotRevs
	return txRevs
}

func checkConflicts(e *env.Env, jobs []*revjob.RevisionJob) error {
	install := make([]*tasks.PackageInstall, 0)
	uninstall := make([]*tasks.PackageUninstall, 0)
	manage := make([]*tasks.FileManage, 0)
	remove := make([]*tasks.FileRemove, 0)
	for _, job := range jobs {
		job.IterTasks(func(t tasks.Task) {
			switch task := t.(type) {
			case *tasks.FileManage:
				manage = append(manage, task)
			case *tasks.FileRemove:
				remove = append(remove, task)
			case *tasks.PackageInstall:
				install = append(install, task)
			case *tasks.PackageUninstall:
				uninstall = append(uninstall, task)
			}
		})
	}
	return tasks.CheckConflicts(e, install, uninstall, remove, manage)
}

func removedFromCurrent(rev *slot.Rev) *slot.Rev {
	return slot.NewRevision(&pb.Rev{
		Id:     rev.Proto().Id,
		Target: pb.RevisionTarget_REMOVED,
		Meta:   rev.Proto().Meta,
		Spec:   rev.Proto().Spec,
	})
}
