package manager

import (
	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/revjob"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks"
	"a.yandex-team.ru/infra/hostctl/pkg/pbutil"
	"a.yandex-team.ru/library/go/core/log"
	"strings"
)

type ExecuteManager struct {
	successfullyRemoved map[string][]string
}

func NewExecuteManager() *ExecuteManager {
	return &ExecuteManager{
		make(map[string][]string),
	}
}

func (m *ExecuteManager) ExecuteSlots(slots map[string]slot.Slot, e *env.Env) {
	slotsJobs := make(map[string]revjob.Set)
	for _, s := range slots {
		jobs := revjob.FromSlot(s)
		for _, j := range jobs {
			revjob.SkipRemovingTasks(j, s.Meta().SkipRemovePhase)
		}
		slotsJobs[s.Name()] = jobs
	}
	// we need to prune jobs
	// each to each slot
	// each to each jobs
	jobs := make(revjob.Set, 0)
	for _, jobsA := range slotsJobs {
		jobs = append(jobs, jobsA...)
	}
	jobs.Prune()
	for _, s := range slots {
		m.Execute(s, slotsJobs[s.Name()], prepareEnv(s.Meta().EnvMode(), s.Name(), e))
	}
}

func prepareEnv(mode slot.EnvMode, name string, e *env.Env) *env.Env {
	e = e.For(name)
	switch mode {
	case "real", "":
		return e
	case "shadow":
		return env.Shadow(e.L)
	case "noop":
		fallthrough
	default: // Unknown/misprinted env better handled with noop
		return env.Noop(e.L)
	}
}

func (m *ExecuteManager) Execute(slot slot.Slot, set revjob.Set, e *env.Env) {
	e.L.Infof("Executing...")
	for _, j := range set {
		p, err := revjob.Plan(j).Fmt(j.SlotName(), j.RevID(), j.Description())
		if err != nil {
			e.L.Errorf("Failed to prettify plan: %s", err.Error())
		} else {
			e.L.Infof("Plan:\n%s", p)
		}
	}
	statuses := set.Execute(e)
	for _, s := range statuses {
		fmted, err := s.Changes.Fmt(slot.Name(), s.ID, s.Description)
		if err != nil {
			e.L.Errorf("Failed to fmt changelog: %s", err.Error())
		} else {
			e.L.Infof("Changelog:\n%s", fmted)
		}
		if s.Err == nil {
			m.successfullyRemoved[slot.Name()] = append(m.successfullyRemoved[slot.Name()], s.ID)
		}
	}
	slot.Status().RestartCount = int32(restartCount(statuses))
	if hasChanges(statuses) {
		e.L.Debugf("Marking '%s' as changed", slot.Name())
		pbutil.TrueCond(slot.Status().Changed, "")
	} else {
		e.L.Debugf("Marking '%s' as not changed", slot.Name())
		pbutil.FalseCond(slot.Status().Changed, "")
	}
	currentErrs, removeErrs := errorsFromStatuses(statuses)
	if len(removeErrs) > 0 && len(currentErrs) > 0 {
		e.L.Debugf("Marking '%s' as  unknown removed/ready because current and removed jobs failed", slot.Name())
		pbutil.UnknownCond(slot.Status().Ready, strings.Join(append(currentErrs, removeErrs...), ";"))
		pbutil.UnknownCond(slot.Status().Removed, strings.Join(append(currentErrs, removeErrs...), ";"))
		return
	}
	if len(removeErrs) > 0 {
		e.L.Debugf("Marking '%s' as not removed, unknown ready because failed to remove", slot.Name())
		pbutil.UnknownCond(slot.Status().Ready, strings.Join(removeErrs, ";"))
		pbutil.FalseCond(slot.Status().Removed, strings.Join(removeErrs, ";"))
		return
	}
	if len(currentErrs) > 0 {
		e.L.Debugf("Marking '%s' as unknown removed and not ready because execution failed", slot.Name())
		pbutil.FalseCond(slot.Status().Ready, strings.Join(currentErrs, ";"))
		pbutil.UnknownCond(slot.Status().Removed, strings.Join(currentErrs, ";"))
		return
	}
	if slot.Current() == nil {
		e.L.Debugf("Marking '%s' as removed, not ready because all jobs finished successfully and current is absent", slot.Name())
		pbutil.TrueCond(slot.Status().Ready, "removed")
		pbutil.TrueCond(slot.Status().Removed, "removed")
	} else {
		e.L.Debugf("Marking '%s' as ready and not removed because successfully executed and current not absent", slot.Name())
		pbutil.TrueCond(slot.Status().Ready, "")
		pbutil.FalseCond(slot.Status().Removed, "")
	}
}

func errorsFromStatuses(statuses []*revjob.JobStatus) (cur []string, rem []string) {
	for _, s := range statuses {
		if s.Err != nil {
			switch s.Description {
			case "removed":
				rem = append(rem, s.Err.Error())
			case "current":
				cur = append(cur, s.Err.Error())
			}
		}
	}
	return
}

func hasChanges(statuses []*revjob.JobStatus) bool {
	for _, s := range statuses {
		if len(s.Changes.Events) > 0 {
			return true
		}
	}
	return false
}

// Very very fragile construction
func restartCount(statuses []*revjob.JobStatus) int {
	c := 0
	for _, s := range statuses {
		for _, l := range s.Changes.Events {
			switch l.Event {
			case "porto.start", "system.service.restart", "system.service.start":
				c++
			}
		}
	}
	return c
}

func (m *ExecuteManager) Plan(slot slot.Slot, l log.Logger) map[string]tasks.Plan {
	jobs := revjob.FromSlot(slot)
	plans := make(map[string]tasks.Plan)
	for _, j := range jobs {
		plan := revjob.Plan(j)
		if len(plan) > 0 {
			plans = map[string]tasks.Plan{j.RevID(): plan}
		} else {
			l.Debugf("Plan(%s): changes not found", slot.Name())
		}
	}
	return plans
}

func (m *ExecuteManager) SuccessfullyRemoved() map[string][]string {
	return m.successfullyRemoved
}

func (m *ExecuteManager) UpdateStatuses(slots map[string]slot.Slot, l log.Logger) {
	for _, s := range slots {
		if !s.Status().IsPending() {
			l.Debugf("Marking '%s' as not pending", s.Name())
			s.Status().SetNotPending()
		}
		// do not set removed status for empty slot if pending=True
		if len(s.Revs()) == 0 && s.Status().IsPending() {
			s.Status().SetReady(s.Status().Pending.Message)
			s.Status().SetNotRemoved()
			continue
		}
		if len(s.Revs()) == 0 {
			l.Debugf("Marking '%s' as removed/ready unk because len(revs) == 0", s.Name())
			const msg = "len(revs) == 0"
			s.Status().SetReady(msg)
			s.Status().SetRemoved(msg)
			continue
		}
	}
}
