package manager

import (
	"fmt"
	"time"

	"a.yandex-team.ru/infra/hostctl/internal/template"

	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/slices"
)

type HostCtl struct {
	*ExecuteManager

	incomingNames []string
	tu            *Target
	Slots         map[string]slot.Slot
	hi            *pb.HostInfo
}

type Target struct {
	units []string
}

func NewTarget() *Target {
	return &Target{}
}

func NewStrictTarget(units []string) *Target {
	return &Target{units: units}
}

func (t *Target) TargetSlots(slots map[string]slot.Slot) map[string]slot.Slot {
	if t.TargetAll() {
		return slots
	}
	targetSlots := make(map[string]slot.Slot)
	for name, s := range slots {
		if slices.ContainsString(t.units, name) {
			targetSlots[name] = s
		}
	}
	return targetSlots
}

func (t *Target) TargetUnits(units map[string]template.Template) map[string]template.Template {
	if t.TargetAll() {
		return units
	}
	targetUnits := make(map[string]template.Template, len(t.units))
	for name, u := range units {
		if slices.ContainsString(t.units, name) {
			targetUnits[name] = u
		}
	}
	return targetUnits
}

func (t *Target) TargetAll() bool {
	return len(t.units) == 0
}

func (t *Target) UnitsNames() []string {
	return t.units
}

func NewHostCtl(info *pb.HostInfo, stateBefore *pb.HostctlState) *HostCtl {
	return &HostCtl{
		ExecuteManager: NewExecuteManager(),
		Slots:          slot.SlotsFromPb(stateBefore),
		hi:             info,
	}
}

func (h *HostCtl) Execute(slots map[string]slot.Slot, e *env.Env) {
	h.ExecuteManager.ExecuteSlots(slots, e)
	h.ExecuteManager.UpdateStatuses(slots, e.L)
}

func (h *HostCtl) ApplyMany(incoming map[string]template.Template, e *env.Env) {
	for name, u := range incoming {
		_, err := h.Apply(u, e)
		if err != nil {
			e.L.Errorf("Failed to apply %s unit: %s", name, err)
		}
	}
}

func (h *HostCtl) UnitsTS() time.Time {
	lastTS := time.Unix(0, 0)
	for _, s := range h.Slots {
		status := s.Status()
		// be this status transition time we understand when unit changed pending status
		ts := status.Pending.TransitionTime.AsTime()
		if lastTS.Before(ts) {
			lastTS = ts
		}
		// be this status transition time we understand when unit changed ready status
		ts = status.Ready.TransitionTime.AsTime()
		if lastTS.Before(ts) {
			lastTS = ts
		}
		// be this status transition time we understand when unit changed version/stage
		ts = status.Changed.TransitionTime.AsTime()
		if lastTS.Before(ts) {
			lastTS = ts
		}
		// be this status transition time we understand when unit was removed
		ts = status.Removed.TransitionTime.AsTime()
		if lastTS.Before(ts) {
			lastTS = ts
		}
	}
	return lastTS
}

func (h *HostCtl) Apply(t template.Template, e *env.Env) (bool, error) {
	l := env.LogWithName(e.L, fmt.Sprintf("[%s]", t.Name()))
	s := h.Slots[t.Name()]
	isNew := s == nil
	// !!! do not remove this code !!!
	// Resetting pending status to False before execution cycle
	// (rendering, validation, throttling, conflicts_resolving)
	// if we'll not reset this to False. we'll never switch pending condition to False
	if !isNew {
		s.ResetPending()
	}
	l.Infof("Rendering...")
	u, err := t.Render(h.hi)
	if err != nil {
		if !isNew {
			l.Infof("Marking as pending: %s", err.Error())
			s.Status().SetPending(err.Error())
		}
		return false, fmt.Errorf("failed to render unit: %w", err)
	}
	// Do not create slot for removed units in state
	// we need to check if absent after rendering
	// because we can render stage=='absent'
	if isNew {
		if u.Absent() {
			l.Infof("Skipping because 'absent'...")
			return false, nil
		} else {
			l.Infof("Creating slot for new unit...")
			s = slot.New(u.Name())
		}
	}
	l.Infof("Validating...")
	if err := u.Valid(); err != nil {
		l.Infof("Marking as pending: %s", err.Error())
		s.Status().SetPending(err.Error())
		return false, fmt.Errorf("validation failed: %w", err)
	}
	// Updating meta on each iteration
	// meta.annotations or others may change after unit loading
	s.UpdateMeta(u.SlotMeta())
	if s.HasChanges(u) {
		e.L.Info("Check conflicts...")
		if err := CheckConflict(e, h.Slots, u); err != nil {
			e.L.Errorf("Found conflict: %s", err)
			s.Status().SetConflicted(err.Error())
			s.Status().SetPending(err.Error())
			return false, err
		}
		s.Status().SetNotConflicted()
		l.Infof("Throttling...")
		if !u.Throttle(e.L, e.Throttler, (*pb.SlotStatus)(s.Status())) {
			l.Errorf("Unit was throttled")
			return false, fmt.Errorf("throttle failed for %s", t.Name())
		}
	} else {
		l.Infof("Skip conflicts check - changes not found...")
		l.Infof("Skip throttling - changes not found...")
	}
	// Appending new slot to state after all steps
	if isNew {
		h.Slots[t.Name()] = s
	}
	l.Infof("Applying revision to slot...")
	s.ApplyUnit(u)
	return s.Changed(), nil
}

// GCSuccessfullyRemovedRevs removes revisions from slot which was successfully executed
func (h *HostCtl) GCSuccessfullyRemovedRevs(l log.Logger) {
	for slotName, s := range h.Slots {
		if h.SuccessfullyRemoved()[slotName] == nil {
			continue
		}
		actualRevs := make([]*slot.Rev, 0)
		// Append current to actual revs to prevent removing by gc.
		// Do not add nil current revision. After removing slot current will nil.
		if c := s.Current(); c != nil {
			actualRevs = append(actualRevs, c)
		}
		// gc only removed revs
		for _, rev := range s.Removed() {
			// filter successfully removed revs
			if slices.ContainsString(h.SuccessfullyRemoved()[slotName], rev.ID()) {
				l.Infof("Removing %s rev@%.11s because it was successfully removed", slotName, rev.ID())
				continue
			}
			actualRevs = append(actualRevs, rev)
		}
		s.SetRevs(actualRevs)
	}
}

// GCRemovedSlots removes slots with Removed=True from state
// We do not need then in hostctl state. we do not want see them in hostctl list|status|show cmds
func (h *HostCtl) GCRemovedSlots(l log.Logger) {
	for name, s := range h.Slots {
		// Slot removal is very rare event.
		// And we need report slot removal.
		// So do not remove slot in 1h
		removalTime := s.Status().Removed.TransitionTime.AsTime()
		if s.Status().IsRemoved() && time.Since(removalTime) > time.Hour {
			l.Infof("Removing %s from state because Removed=True", name)
			delete(h.Slots, name)
		}
	}
}
