package manager

import (
	"errors"
	"testing"

	"a.yandex-team.ru/infra/hostctl/internal/template"

	"github.com/golang/protobuf/ptypes"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"

	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/env/pacman"
	"a.yandex-team.ru/infra/hostctl/pkg/pbutil"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
)

func testHostInfo() *pb.HostInfo {
	return &pb.HostInfo{
		Hostname:     "man1-3720.search.yandex.net",
		Num:          17,
		WalleProject: "rtc-prestable",
		WalleTags: []string{
			"rtc", "rtc.automation-enabled", "rtc.cohabitation-enabled", "rtc.gpu-none", "rtc.reboot_segment-prestable-def",
			"rtc.scheduler-gencfg", "rtc.stage-prestable", "rtc_network", "runtime", "search", "skynet_installed", "yasm_monitored"},
		NetSwitch: "man1-s66",
		GencfgGroups: []string{
			"ALL_RTC", "ALL_RUNTIME", "ALL_SEARCH", "MAN_JUGGLER_CLIENT_STABLE", "MAN_KERNEL_UPDATE_3", "MAN_RTC_SLA_TENTACLES_PROD", "MAN_RUNTIME",
			"MAN_SAAS_CLOUD", "MAN_SAAS_CLOUD_BLACKLISTED_REFRESH", "MAN_SAAS_CLOUD_SHMICK", "MAN_SEARCH", "MAN_YASM_YASMAGENT_STABLE"},
		Location:      "man",
		Dc:            "man",
		KernelRelease: "4.19.138-35",
	}
}

func TestHostCtl_Apply_successWithNewUnit(t *testing.T) {
	l, _ := zap.New(zap.TSKVConfig(log.DebugLevel))
	e := env.Noop(l)
	e.Throttler = &okThrottler{}
	e.Pacman = okPacman()
	h := NewHostCtl(testHostInfo(), &pb.HostctlState{
		UnitsTs: ptypes.TimestampNow(),
	})
	revIDAfter, _ := pbutil.PbDigest(&pb.PackageSetSpec{
		Packages: make([]*pb.SystemPackage, 0),
		Files:    make([]*pb.ManagedFile, 0),
	})
	changed, err := h.Apply(template.NewTestTemplate(
		&pb.ObjectMeta{
			Kind:            "PackageSet",
			Name:            "test-package-set",
			Version:         "testing",
			DeleteRequested: false,
			Labels:          make(map[string]string),
			Annotations:     map[string]string{"stage": "testing"},
			Ctx:             &pb.Context{},
		},
		&pb.PackageSetSpec{
			Packages: make([]*pb.SystemPackage, 0),
			Files:    make([]*pb.ManagedFile, 0),
		}), e)
	if err != nil {
		t.Error(err)
	}
	assert.True(t, changed)
	slots := h.Slots
	if len(slots) != 1 {
		t.Error("len(h.Slots()) should be 1")
		return
	}
	s, ok := slots["test-package-set"]
	if !ok {
		t.Error("applied slot not found in state")
		return
	}
	assert.Equal(t, "False", s.Status().Throttled.Status)
	assert.Equal(t, revIDAfter, s.Current().ID())
}

func TestHostCtl_Apply_successWithChanged(t *testing.T) {
	l, _ := zap.New(zap.TSKVConfig(log.DebugLevel))
	e := env.Noop(l)
	e.Throttler = &okThrottler{}
	e.Pacman = okPacman()
	meta := &pb.ObjectMeta{
		Kind:            "PackageSet",
		Name:            "test-package-set",
		Version:         "testing",
		DeleteRequested: false,
		Labels:          make(map[string]string),
		Annotations:     map[string]string{"stage": "testing"},
		Ctx:             &pb.Context{},
	}
	rm := &pb.RevisionMeta{
		Version: "testing",
	}
	sm := &pb.SlotMeta{
		Labels:      make(map[string]string),
		Annotations: map[string]string{"stage": "testing"},
	}
	spec := &pb.PackageSetSpec{
		Packages: []*pb.SystemPackage{{
			Name:    "p",
			Version: "1",
		}},
		Files: []*pb.ManagedFile{{
			Path:    "/tmp/a",
			Content: "a",
			User:    "root",
			Group:   "root",
			Mode:    "0644",
		}},
	}
	revID, _ := pbutil.PbDigest(spec)
	h := NewHostCtl(testHostInfo(), &pb.HostctlState{
		UnitsTs: ptypes.TimestampNow(),
		Slots: []*pb.Slot{{
			Name:   "test-package-set",
			Status: emptyPbStatus(),
			Revs: []*pb.Rev{{
				Id:     revID,
				Target: pb.RevisionTarget_CURRENT,
				Meta:   rm,
				Spec:   &pb.Rev_PackageSet{PackageSet: spec},
			}},
			Meta: sm,
		}},
	})
	revIDAfter, _ := pbutil.PbDigest(&pb.PackageSetSpec{
		Packages: make([]*pb.SystemPackage, 0),
		Files:    make([]*pb.ManagedFile, 0),
	})
	changed, err := h.Apply(template.NewTestTemplate(meta,
		&pb.PackageSetSpec{
			Packages: make([]*pb.SystemPackage, 0),
			Files:    make([]*pb.ManagedFile, 0),
		}), e)
	if err != nil {
		t.Error(err)
	}
	assert.True(t, changed)
	slots := h.Slots
	if len(slots) != 1 {
		t.Error("len(h.Slots()) should be 1")
		return
	}
	s, ok := slots["test-package-set"]
	if !ok {
		t.Error("applied slot not found in state")
		return
	}
	assert.Equal(t, "False", s.Status().Throttled.Status)
	assert.Equal(t, revIDAfter, s.Current().ID())
}

func TestHostCtl_Apply_throttleFailWithChanged(t *testing.T) {
	l, _ := zap.New(zap.TSKVConfig(log.DebugLevel))
	e := env.Noop(l)
	e.Throttler = &failThrottler{}
	e.Pacman = okPacman()
	meta := &pb.ObjectMeta{
		Kind:            "PackageSet",
		Name:            "test-package-set",
		Version:         "testing",
		DeleteRequested: false,
		Labels:          make(map[string]string),
		Annotations:     map[string]string{"stage": "testing"},
		Ctx:             &pb.Context{},
	}
	rm := &pb.RevisionMeta{
		Version: "testing",
	}
	sm := &pb.SlotMeta{
		Labels:      make(map[string]string),
		Annotations: map[string]string{"stage": "testing"},
	}
	spec := &pb.PackageSetSpec{
		Packages: []*pb.SystemPackage{{
			Name:    "p",
			Version: "1",
		}},
		Files: []*pb.ManagedFile{{
			Path:    "/tmp/a",
			Content: "a",
			User:    "root",
			Group:   "root",
			Mode:    "0644",
		}},
	}
	revID, _ := pbutil.PbDigest(spec)
	h := NewHostCtl(testHostInfo(), &pb.HostctlState{
		UnitsTs: ptypes.TimestampNow(),
		Slots: []*pb.Slot{{
			Name:   "test-package-set",
			Status: emptyPbStatus(),
			Revs: []*pb.Rev{{
				Id:     revID,
				Target: pb.RevisionTarget_CURRENT,
				Meta:   rm,
				Spec:   &pb.Rev_PackageSet{PackageSet: spec},
			}},
			Meta: sm,
		}},
	})
	changed, err := h.Apply(template.NewTestTemplate(meta,
		&pb.PackageSetSpec{
			Packages: make([]*pb.SystemPackage, 0),
			Files:    make([]*pb.ManagedFile, 0),
		}), e)
	if err != nil {
		assert.EqualError(t, err, "throttle failed for test-package-set")
	} else {
		t.Error("should return throttle fail")
	}
	assert.False(t, changed)
	slots := h.Slots
	if len(slots) != 1 {
		t.Error("len(h.Slots()) should be 1")
		return
	}
	s, ok := slots["test-package-set"]
	if !ok {
		t.Error("applied slot not found in state")
		return
	}
	assert.Equal(t, "True", s.Status().Throttled.Status)
	assert.Equal(t, "True", s.Status().Pending.Status)
	assert.Equal(t, revID, s.Current().ID())
}

func TestHostCtl_Apply_conflictsWithChanged(t *testing.T) {
	l, _ := zap.New(zap.TSKVConfig(log.DebugLevel))
	e := env.Noop(l)
	e.Throttler = &okThrottler{}
	e.Pacman = failPacman()
	meta := &pb.ObjectMeta{
		Kind:            "PackageSet",
		Name:            "test-package-set",
		Version:         "testing",
		DeleteRequested: false,
		Labels:          make(map[string]string),
		Annotations:     map[string]string{"stage": "testing"},
		Ctx:             &pb.Context{},
	}
	rm := &pb.RevisionMeta{
		Version: "testing",
	}
	sm := &pb.SlotMeta{
		Labels:      make(map[string]string),
		Annotations: map[string]string{"stage": "testing"},
	}
	spec := &pb.PackageSetSpec{
		Packages: []*pb.SystemPackage{{
			Name:    "p",
			Version: "1",
		}},
		Files: []*pb.ManagedFile{{
			Path:    "/tmp/a",
			Content: "a",
			User:    "root",
			Group:   "root",
			Mode:    "0644",
		}},
	}
	revID, _ := pbutil.PbDigest(spec)
	h := NewHostCtl(testHostInfo(), &pb.HostctlState{
		UnitsTs: ptypes.TimestampNow(),
		Slots: []*pb.Slot{{
			Name:   "test-package-set",
			Status: emptyPbStatus(),
			Revs: []*pb.Rev{{
				Id:     revID,
				Target: pb.RevisionTarget_CURRENT,
				Meta:   rm,
				Spec:   &pb.Rev_PackageSet{PackageSet: spec},
			}},
			Meta: sm,
		}},
	})
	changed, err := h.Apply(template.NewTestTemplate(meta,
		&pb.PackageSetSpec{
			Packages: make([]*pb.SystemPackage, 0),
			Files:    make([]*pb.ManagedFile, 0),
		}), e)
	if err != nil {
		assert.EqualError(t, err, "mock err")
	} else {
		t.Errorf("apt install --dry run should fail Apply with conflict err")
	}
	assert.False(t, changed)
	slots := h.Slots
	if len(slots) != 1 {
		t.Error("len(h.Slots()) should be 1")
		return
	}
	s, ok := slots["test-package-set"]
	if !ok {
		t.Error("applied slot not found in state")
		return
	}
	assert.Equal(t, "True", s.Status().Conflicted.Status)
	assert.Equal(t, "True", s.Status().Pending.Status)
	assert.Equal(t, revID, s.Current().ID())
}

func TestHostCtl_Apply_validationFailWithChanged(t *testing.T) {
	l, _ := zap.New(zap.TSKVConfig(log.DebugLevel))
	e := env.Noop(l)
	e.Throttler = &okThrottler{}
	e.Pacman = okPacman()
	meta := &pb.ObjectMeta{
		Kind:            "PackageSet",
		Name:            "test-package-set",
		Version:         "testing",
		DeleteRequested: false,
		Labels:          make(map[string]string),
		Annotations:     map[string]string{"stage": "testing"},
		Ctx:             &pb.Context{},
	}
	rm := &pb.RevisionMeta{
		Version: "testing",
	}
	sm := &pb.SlotMeta{
		Labels:      make(map[string]string),
		Annotations: map[string]string{"stage": "testing"},
	}
	spec := &pb.PackageSetSpec{
		Packages: []*pb.SystemPackage{{
			Name:    "p",
			Version: "1",
		}},
		Files: []*pb.ManagedFile{{
			Path:    "/tmp/a",
			Content: "a",
			User:    "root",
			Group:   "root",
			Mode:    "0644",
		}},
	}
	revID, _ := pbutil.PbDigest(spec)
	h := NewHostCtl(testHostInfo(), &pb.HostctlState{
		UnitsTs: ptypes.TimestampNow(),
		Slots: []*pb.Slot{{
			Name:   "test-package-set",
			Status: emptyPbStatus(),
			Revs: []*pb.Rev{{
				Id:     revID,
				Target: pb.RevisionTarget_CURRENT,
				Meta:   rm,
				Spec:   &pb.Rev_PackageSet{PackageSet: spec},
			}},
			Meta: sm,
		}},
	})
	changed, err := h.Apply(template.NewTestTemplate(meta,
		&pb.PackageSetSpec{
			Packages: []*pb.SystemPackage{{
				Name:    "p",
				Version: "",
			}},
			Files: make([]*pb.ManagedFile, 0),
		}), e)
	if err != nil {
		assert.EqualError(t, err, "validation failed: packages[0]: p version: empty version")
	} else {
		t.Error("should return throttle fail")
	}
	assert.False(t, changed)
	slots := h.Slots
	if len(slots) != 1 {
		t.Error("len(h.Slots()) should be 1")
		return
	}
	s, ok := slots["test-package-set"]
	if !ok {
		t.Error("applied slot not found in state")
		return
	}
	assert.Equal(t, "True", s.Status().Pending.Status)
	assert.Equal(t, revID, s.Current().ID())
}

func TestHostCtl_Apply_successWithNotChanged(t *testing.T) {
	l, _ := zap.New(zap.TSKVConfig(log.DebugLevel))
	e := env.Noop(l)
	e.Throttler = &okThrottler{}
	e.Pacman = okPacman()
	meta := &pb.ObjectMeta{
		Kind:            "PackageSet",
		Name:            "test-package-set",
		Version:         "testing",
		DeleteRequested: false,
		Labels:          make(map[string]string),
		Annotations:     map[string]string{"stage": "testing"},
		Ctx:             &pb.Context{},
	}
	rm := &pb.RevisionMeta{
		Version: "testing",
	}
	sm := &pb.SlotMeta{
		Labels:      make(map[string]string),
		Annotations: map[string]string{"stage": "testing"},
	}
	spec := &pb.PackageSetSpec{
		Packages: []*pb.SystemPackage{{
			Name:    "p",
			Version: "1",
		}},
		Files: []*pb.ManagedFile{{
			Path:    "/tmp/a",
			Content: "a",
			User:    "root",
			Group:   "root",
			Mode:    "0644",
		}},
	}
	revID, _ := pbutil.PbDigest(spec)
	h := NewHostCtl(testHostInfo(), &pb.HostctlState{
		UnitsTs: ptypes.TimestampNow(),
		Slots: []*pb.Slot{{
			Name:   "test-package-set",
			Status: emptyPbStatus(),
			Revs: []*pb.Rev{{
				Id:     revID,
				Target: pb.RevisionTarget_CURRENT,
				Meta:   rm,
				Spec:   &pb.Rev_PackageSet{PackageSet: spec},
			}},
			Meta: sm,
		}},
	})
	changed, err := h.Apply(template.NewTestTemplate(meta, spec), e)
	if err != nil {
		t.Error(err)
	}
	assert.False(t, changed)
	slots := h.Slots
	if len(slots) != 1 {
		t.Error("len(h.Slots()) should be 1")
		return
	}
	s, ok := slots["test-package-set"]
	if !ok {
		t.Error("applied slot not found in state")
		return
	}
	assert.Equal(t, revID, s.Current().ID())
}

type okThrottler struct {
}

func (*okThrottler) Can(string, map[string]string) error {
	return nil
}

type failThrottler struct {
}

func (*failThrottler) Can(string, map[string]string) error {
	return errors.New("throttle fail")
}

func okPacman() pacman.PackageManager {
	p := pacman.NewDPKGMock()
	p.On("InstallSetDryRun", mock.Anything).Return(nil)
	p.On("InstallSet", mock.Anything).Return(nil)
	return p
}

func failPacman() pacman.PackageManager {
	p := pacman.NewDPKGMock()
	p.On("InstallSetDryRun", mock.Anything).Return(errors.New("mock err"))
	p.On("InstallSet", mock.Anything).Return(errors.New("mock err"))
	return p
}

func emptyPbStatus() *pb.SlotStatus {
	s := &pb.SlotStatus{}
	slot.NormalizeStatus((*slot.Status)(s))
	return s
}
