package report

import (
	"a.yandex-team.ru/infra/hmserver/pkg/reporter/client"
	reporterpb "a.yandex-team.ru/infra/hmserver/proto"
	"a.yandex-team.ru/infra/hostctl/internal/slot"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/core/buildinfo"
	"a.yandex-team.ru/library/go/core/log"
	"github.com/golang/protobuf/ptypes"
	"strings"
	"time"
)

func NewReporter(client *client.Reporter, config *Config, l log.Logger, ctype string) Reporter {
	return &hmReporter{r: client, config: config, l: l, ctype: ctype}
}

func NewNoopReporter() Reporter {
	return &noopHMReporter{}
}

type hmReporter struct {
	config *Config
	r      *client.Reporter
	l      log.Logger
	ctype  string
}

func (rs *hmReporter) Report(slots map[string]slot.Slot, include *pb.HostInfo, unitsTS time.Time) error {
	reports := make([]*reporterpb.Unit, 0)
	for name, slot := range slots {
		if st := slot.Status(); st.IsRemoved() && !st.IsPending() {
			continue
		}
		rs.l.Infof("Reporting '%s' to hm-reporter...", slot.Name())
		r := &reporterpb.Unit{
			Node:           rs.config.Hostname,
			Ready:          getReportStatus(slot.Status().Ready),
			Pending:        getReportStatus(slot.Status().Pending),
			LastTransition: slot.Status().Changed.TransitionTime,
			Name:           name,
			Stage:          string(slot.Meta().Stage()),
			// overriding after with version of current revision
			Version: "-",
		}
		if c := slot.Current(); c != nil {
			r.Version = c.Meta().Version
			r.Kind = string(slot.Kind())
		}
		reports = append(reports, r)
	}
	reports = append(reports, rs.collectHostctl(buildinfo.Info))
	unitsTSProto, _ := ptypes.TimestampProto(unitsTS)
	return rs.r.SendReport(reports, include, unitsTSProto)
}

const buildInfoTimeLayout = "2006-01-02T15:04:05"

func (rs *hmReporter) collectHostctl(info buildinfo.BuildInfo) *reporterpb.Unit {
	// info.Date = '2021-01-14T12:35:06.000000Z'
	short := strings.Split(info.Date, ".")[0] // 2021-01-14T12:35:06
	buildDate, _ := time.Parse(buildInfoTimeLayout, short)
	buildDateProto, _ := ptypes.TimestampProto(buildDate)
	return &reporterpb.Unit{
		Node:           rs.config.Hostname,
		LastTransition: buildDateProto,
		Name:           "hostctl",
		Kind:           "Hostman",
		Stage:          rs.ctype, // ["experiment", "prestable", "production", "core"]
		Version:        info.SVNRevision,
		// stub
		Ready:   reporterpb.Status_TRUE,
		Pending: reporterpb.Status_FALSE,
	}
}

func (rs *hmReporter) Description() string {
	return "hm-reporter"
}

type noopHMReporter struct {
}

func (rs *noopHMReporter) Report(slots map[string]slot.Slot, include *pb.HostInfo, unitsTS time.Time) error {
	return nil
}

func (rs *noopHMReporter) Description() string {
	return "noop-hm-reporter"
}

func getReportStatus(status *pb.Condition) reporterpb.Status {
	switch status.Status {
	case "True":
		return reporterpb.Status_TRUE
	case "False":
		return reporterpb.Status_FALSE
	default:
		return reporterpb.Status_UNKNOWN
	}
}
