package report

import (
	"context"
	"fmt"
	"strings"
	"time"

	"a.yandex-team.ru/infra/hostctl/internal/juggler"
	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/unit/kind"
	"a.yandex-team.ru/library/go/core/log"

	pb "a.yandex-team.ru/infra/hostctl/proto"
)

func NewJugglerReporter(l log.Logger, c *Config) Reporter {
	return &jugglerReporter{
		c:  c,
		jc: juggler.NewClient(),
		l:  l,
	}
}

type jugglerReporter struct {
	c  *Config
	l  log.Logger
	jc *juggler.Client
}

func conditionToJuggler(cond *pb.Condition) string {
	switch cond.Status {
	case "True":
		return "OK"
	case "False":
		return "CRIT"
	case "Unknown":
		return "CRIT"
	default:
		return ""
	}
}

func extractPDChecks(slots map[string]slot.Slot) []juggler.Event {
	signals := make([]juggler.Event, 0)
	for name, s := range slots {
		k := s.Kind()
		// Skipping not porto daemons
		if k != kind.PortoDaemon {
			continue
		}
		// Skip sending signals if slot was removed (stage=absent)
		if s.Status().IsRemoved() && !s.Status().IsPending() {
			continue
		}
		if running := conditionToJuggler(s.Status().Running); running != "" {
			signals = append(signals, juggler.MakeWalle(
				fmt.Sprintf("%s-%s-running", k.Kebab(), name),
				running,
				s.Status().Running.Message))
		}
	}
	return signals
}

func unitsStatusToJugglerSignals(slots map[string]slot.Slot) []juggler.Event {
	signals := make([]juggler.Event, 0)
	for _, s := range slots {
		k := s.Kind()
		// Skip sending signals if slot was removed (stage=absent)
		if s.Status().IsRemoved() && !s.Status().IsPending() {
			continue
		}
		if ready := conditionToJuggler(s.Status().Ready); ready != "" {
			// Strip trailing '@' from slot name as juggler does not allow it
			name := strings.TrimRight(s.Name(), "@")
			// Replace '@' with '-' as juggler does not allow it
			name = strings.ReplaceAll(name, "@", "-")
			signals = append(signals, juggler.MakeWalle(
				fmt.Sprintf("%s-%s-ready", k.Kebab(), name),
				ready,
				s.Status().Ready.Message))
		}
	}
	return signals
}

func ExtractJuggler(slots map[string]slot.Slot) []juggler.Event {
	checks := unitsStatusToJugglerSignals(slots)
	checks = append(checks, extractPDChecks(slots)...)
	return checks
}

func (rep *jugglerReporter) Report(slots map[string]slot.Slot, _ *pb.HostInfo, _ time.Time) error {
	checks := ExtractJuggler(slots)
	checksToSend := make([]juggler.Event, len(checks))
	for i, check := range checks {
		checksToSend[i] = check.WithHostname(rep.c.Hostname)
	}
	for _, v := range checksToSend {
		rep.l.Infof("Check %s: %s...", v.Service, v.Status)
	}
	ctx, cancel := context.WithTimeout(context.Background(), reportTimeout)
	defer cancel()
	err := rep.jc.PushEvents(ctx, juggler.Request{
		Events: checksToSend,
	})
	if err != nil {
		return fmt.Errorf("failed to send metrics: %w", err)
	}
	return nil
}

func (rep *jugglerReporter) Description() string {
	return "juggler-reporter"
}

func NewNoopJugglerReporter(log.Logger, *Config) Reporter {
	return &noopJugglerReporter{}
}

type noopJugglerReporter struct {
}

func (rep *noopJugglerReporter) Report(slots map[string]slot.Slot, _ *pb.HostInfo, _ time.Time) error {
	return nil
}

func (rep *noopJugglerReporter) Description() string {
	return "noop-juggler-reporter"
}
