package report

import (
	"testing"

	"a.yandex-team.ru/infra/hostctl/internal/juggler"
	"a.yandex-team.ru/infra/hostctl/internal/slot"
	pb "a.yandex-team.ru/infra/hostctl/proto"
)

func TestExtractJuggler(t *testing.T) {
	type args struct {
		slots map[string]slot.Slot
	}
	tests := []struct {
		name string
		args args
		want []juggler.Event
	}{{
		name: "PortoDaemon",
		args: args{
			slots: map[string]slot.Slot{"juggler-agent-rtc": slot.NewSlot(&pb.Slot{
				Name: "juggler-agent-rtc",
				Revs: []*pb.Rev{{
					Target: pb.RevisionTarget_CURRENT,
					Meta: &pb.RevisionMeta{
						Kind: "PortoDaemon",
					},
				}},
				Status: &pb.SlotStatus{
					Pending:      falseCond(),
					Ready:        trueCond(),
					Removed:      falseCond(),
					Throttled:    falseCond(),
					Changed:      falseCond(),
					Conflicted:   falseCond(),
					Installed:    trueCond(),
					Running:      trueCond(),
					RestartCount: 0,
					RespawnCount: 0,
				},
				Meta: &pb.SlotMeta{},
			})},
		},
		want: []juggler.Event{juggler.MakeWalle(
			"porto-daemon-juggler-agent-rtc-running",
			"OK",
			"True"), juggler.MakeWalle(
			"porto-daemon-juggler-agent-rtc-ready",
			"OK",
			"True")},
	}, {
		name: "PackageSet",
		args: args{
			slots: map[string]slot.Slot{"upstream-packages": slot.NewSlot(&pb.Slot{
				Name: "upstream-packages",
				Revs: []*pb.Rev{{
					Target: pb.RevisionTarget_CURRENT,
					Meta: &pb.RevisionMeta{
						Kind: "PackageSet",
					},
				}},
				Status: &pb.SlotStatus{
					Pending:      falseCond(),
					Ready:        trueCond(),
					Removed:      falseCond(),
					Throttled:    falseCond(),
					Changed:      trueCond(),
					Conflicted:   falseCond(),
					Installed:    trueCond(),
					Running:      trueCond(),
					RestartCount: 0,
				},
				Meta: &pb.SlotMeta{},
			})},
		},
		want: []juggler.Event{juggler.MakeWalle(
			"package-set-upstream-packages-ready",
			"OK",
			"True")},
	}, {
		name: "removed",
		args: args{
			slots: map[string]slot.Slot{"upstream-packages": slot.NewSlot(&pb.Slot{
				Name: "upstream-packages",
				Revs: []*pb.Rev{{
					Target: pb.RevisionTarget_CURRENT,
					Meta: &pb.RevisionMeta{
						Kind: "PackageSet",
					},
				}},
				Status: &pb.SlotStatus{
					Pending:    falseCond(),
					Ready:      trueCond(),
					Removed:    trueCond(),
					Throttled:  falseCond(),
					Changed:    trueCond(),
					Conflicted: falseCond(),
					Installed:  trueCond(),
					Running:    trueCond(),
				},
				Meta: &pb.SlotMeta{},
			})},
		},
		want: make([]juggler.Event, 0),
	}, {
		name: "Templated",
		args: args{
			slots: map[string]slot.Slot{"yandex-sol-rtc@": slot.NewSlot(&pb.Slot{
				Name: "yandex-sol-rtc@",
				Revs: []*pb.Rev{{
					Target: pb.RevisionTarget_CURRENT,
					Meta: &pb.RevisionMeta{
						Kind: "SystemService",
					},
				}},
				Status: &pb.SlotStatus{
					Pending:      falseCond(),
					Ready:        trueCond(),
					Removed:      falseCond(),
					Throttled:    falseCond(),
					Changed:      falseCond(),
					Conflicted:   falseCond(),
					Installed:    trueCond(),
					Running:      trueCond(),
					RestartCount: 0,
					RespawnCount: 0,
				},
				Meta: &pb.SlotMeta{},
			})},
		},
		want: []juggler.Event{juggler.MakeWalle(
			"system-service-yandex-sol-rtc-ready",
			"OK",
			"True")},
	},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			j := ExtractJuggler(tt.args.slots)
			if len(j) != len(tt.want) {
				t.Errorf("ExtractMonitoring() got = %v, want %v", j, tt.want)
			} else {
				for i := 0; i < len(j); i++ {
					if !jugglerChecksEqual(j[i], tt.want[i]) {
						t.Errorf("ExtractMonitoring() got = %v, want %v", j, tt.want)
					}
				}
			}
		})
	}
}

func jugglerChecksEqual(a, b juggler.Event) bool {
	return a.Status == b.Status
}
