package report

import (
	"a.yandex-team.ru/infra/hmserver/pkg/yasmclient"
	"a.yandex-team.ru/infra/hostctl/internal/unit/kind"

	"context"
	"fmt"
	"sort"
	"strings"
	"time"

	"a.yandex-team.ru/infra/hostctl/internal/juggler"
	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/library/go/core/log"

	pb "a.yandex-team.ru/infra/hostctl/proto"
)

const (
	reportTimeout = 5 * time.Second
	signalTTL     = 30 * 60 // Seconds
)

type YasmVisitor interface {
	VisitUnit(*slot.Status)
	VisitPortoDaemon(*slot.Status)
	VisitSystemService(*slot.Status)
	Values() []yasmclient.YasmValue
	Reset()
}

func NewYasmVisitor(rebooted bool) YasmVisitor {
	return &yasmVisitor{rebooted: rebooted, values: make([]yasmclient.YasmValue, 0)}
}

type yasmVisitor struct {
	rebooted bool
	values   []yasmclient.YasmValue
}

// Reset discards any accumulated metric values, making visitor ready to be reused
func (v *yasmVisitor) Reset() {
	v.values = make([]yasmclient.YasmValue, 0)
}

// VisitUnit collect common values for all units
func (v *yasmVisitor) VisitUnit(s *slot.Status) {
	// https://wiki.yandex-team.ru/golovan/userdocs/datatypes/#histogram
	// Histogram bounds are powers of 1.5
	// [..., 0.66, 1, 1.5, 2.25, ...]
	// Our scheme is:
	// 1 - ready ([-inf, 0.66] bucket)
	// 0 - not ready ([0.66, 1.5] bucket)
	// 2 - special value for enforce graphics cleaning after removal ([1.5, inf] bucket)
	unitReady := 0
	if s.IsReady() {
		unitReady = 1
	}
	if s.IsRemoved() {
		unitReady = 2
	}
	unitPending := 0
	if s.IsPending() {
		unitPending = 1
	}
	unitChanged := 0
	// HOSTMAN-978
	if s.IsChanged() && !v.rebooted {
		unitChanged = 1
	}
	v.observe("unit_ready_thhh", unitReady)
	v.observe("unit_pending_thhh", unitPending)
	v.observe("unit_changed_thhh", unitChanged)
}

// VisitPortoDaemon collect PortoDaemon specific values
func (v *yasmVisitor) VisitPortoDaemon(s *slot.Status) {
	// https://wiki.yandex-team.ru/golovan/userdocs/datatypes/#histogram
	// Histogram bounds are powers of 1.5
	// [..., 0.66, 1, 1.5, 2.25, ...]
	// Our scheme is:
	// 1 - running ([-inf, 0.66] bucket)
	// 0 - not running ([0.66, 1.5] bucket)
	// 2 - special value for enforce graphics cleaning after removal ([1.5, inf] bucket)
	isRunningSignal := 0
	if s.IsRunning() {
		isRunningSignal = 1
	}
	if s.IsRemoved() {
		isRunningSignal = 2
	}
	v.observe("unit_running_thhh", isRunningSignal)
	v.observe("unit_restart_count_thhh", int(s.RestartCount))
	v.observe("unit_respawn_count_thhh", int(s.RespawnCount))
}

// VisitSystemService collect SystemService specific values
func (v *yasmVisitor) VisitSystemService(s *slot.Status) {
	// https://wiki.yandex-team.ru/golovan/userdocs/datatypes/#histogram
	// Histogram bounds are powers of 1.5
	// [..., 0.66, 1, 1.5, 2.25, ...]
	// Our scheme is:
	// 1 - running ([-inf, 0.66] bucket)
	// 0 - not running ([0.66, 1.5] bucket)
	// 2 - special value for enforce graphics cleaning after removal ([1.5, inf] bucket)
	isRunningSignal := 0
	if s.IsRunning() {
		isRunningSignal = 1
	}
	if s.IsRemoved() {
		isRunningSignal = 2
	}
	v.observe("unit_running_thhh", isRunningSignal)
	v.observe("unit_restart_count_thhh", int(s.RestartCount))
}

func (v *yasmVisitor) observe(name string, value interface{}) {
	v.values = append(v.values, yasmclient.YasmValue{Name: name, Value: value})
}

func (v *yasmVisitor) Values() []yasmclient.YasmValue {
	return v.values
}

type Tags map[string]string

func (t Tags) String() string {
	b := strings.Builder{}
	n := len(t) - 1
	for k, v := range t {
		b.WriteString(k)
		b.WriteByte('=')
		b.WriteString(v)
		if n != 0 {
			b.WriteByte(' ')
			n--
		}
	}
	return b.String()
}

type YasmReporterBuilder func(tags Tags, l log.Logger, c *Config, rebooted bool) Reporter

func NewYasmReporter(tags Tags, l log.Logger, c *Config, rebooted bool) Reporter {
	// default tags
	ts := Tags{
		"itype": "runtimecloud",
		"ctype": "production",
	}
	for k, v := range tags {
		ts[k] = v
	}
	return &yasmReporter{
		tags:     ts,
		c:        c,
		yc:       yasmclient.NewLocal(),
		jc:       juggler.NewClient(),
		l:        l,
		rebooted: rebooted,
	}
}

type yasmReporter struct {
	tags     Tags
	c        *Config
	l        log.Logger
	yc       *yasmclient.YasmClient
	jc       *juggler.Client
	rebooted bool // do not send changed true when we running first time after reboot (HOSTMAN-978)
}

func makeSignalTags(commonTags Tags, kebab, name string) Tags {
	signalTags := make(Tags)
	for k, v := range commonTags {
		signalTags[k] = v
	}
	signalTags["unit"] = name
	signalTags["unit_type"] = kebab
	return signalTags
}

func ExtractYASM(tags Tags, slots map[string]slot.Slot, rebooted bool) []yasmclient.YasmMetrics {
	metrics := make([]yasmclient.YasmMetrics, 0)
	v := NewYasmVisitor(rebooted)
	// Sort names to collect/send monitoring reproducible.
	// In same order all times.
	// Make easier to test and read logs.
	names := make([]string, 0, len(slots))
	for name := range slots {
		names = append(names, name)
	}
	sort.Strings(names)
	for _, name := range names {
		s := slots[name]
		v.Reset() // clear old metrics
		v.VisitUnit(s.Status())
		// Append kind specific values
		switch s.Kind() {
		case kind.PortoDaemon:
			v.VisitPortoDaemon(s.Status())
		case kind.SystemService:
			v.VisitSystemService(s.Status())
		}
		metrics = append(metrics, yasmclient.YasmMetrics{
			Tags:   makeSignalTags(tags, s.Kind().Kebab(), name),
			TTL:    signalTTL,
			Values: v.Values(),
		})
	}
	return metrics
}

func (rep *yasmReporter) Report(slots map[string]slot.Slot, _ *pb.HostInfo, _ time.Time) error {
	signals := ExtractYASM(rep.tags, slots, rep.rebooted)
	rep.l.Infof("Yasm common tags: %s", rep.tags)
	for _, v := range signals {
		for _, s := range v.Values {
			rep.l.Debugf("Tags: %s", v.Tags)
			rep.l.Infof("%s %s %s=%d", v.Tags["unit_type"], v.Tags["unit"], s.Name, s.Value)
		}
	}
	ctx, cancel := context.WithTimeout(context.Background(), reportTimeout)
	defer cancel()
	err := rep.yc.SendMetrics(ctx, signals)
	if err != nil {
		return fmt.Errorf("failed to send metrics: %w", err)
	}
	return nil
}

func (rep *yasmReporter) Description() string {
	return "yasm-reporter"
}

func NewNoopYasmReporter(Tags, log.Logger, *Config, bool) Reporter {
	return &noopYasmReporter{}
}

type noopYasmReporter struct {
}

func (rep *noopYasmReporter) Report(_ map[string]slot.Slot, _ *pb.HostInfo, _ time.Time) error {
	return nil
}

func (rep *noopYasmReporter) Description() string {
	return "noop-yasm-reporter"
}
