package report

import (
	"fmt"
	"testing"

	"a.yandex-team.ru/infra/hmserver/pkg/yasmclient"
	"github.com/golang/protobuf/ptypes"
	"github.com/stretchr/testify/assert"

	"a.yandex-team.ru/infra/hostctl/internal/slot"
	pb "a.yandex-team.ru/infra/hostctl/proto"
)

func TestTags_String(t *testing.T) {
	tags := Tags{
		"ctype": "production",
		"prj":   "yt-sas-hahn",
	}
	v := tags.String()
	// Keys iteration in a map is non-deterministic
	if v != "ctype=production prj=yt-sas-hahn" && v != "prj=yt-sas-hahn ctype=production" {
		t.Errorf("tags.String() failed: %s", tags)
	}
}

func TestExtractMonitoring(t *testing.T) {
	type args struct {
		tags     Tags
		slots    map[string]slot.Slot
		rebooted bool
	}
	tests := []struct {
		name string
		args args
		want []yasmclient.YasmMetrics
	}{
		{
			name: "PortoDaemon",
			args: args{
				tags: Tags{"ctype": "prod"},
				slots: map[string]slot.Slot{"juggler-agent-rtc": slot.NewSlot(&pb.Slot{
					Name: "juggler-agent-rtc",
					Revs: []*pb.Rev{{
						Target: pb.RevisionTarget_CURRENT,
						Meta: &pb.RevisionMeta{
							Kind: "PortoDaemon",
						},
					}},
					Status: &pb.SlotStatus{
						Pending:      falseCond(),
						Ready:        trueCond(),
						Removed:      falseCond(),
						Throttled:    falseCond(),
						Changed:      falseCond(),
						Conflicted:   falseCond(),
						Installed:    trueCond(),
						Running:      trueCond(),
						RestartCount: 0,
						RespawnCount: 0,
					},
					Meta: &pb.SlotMeta{},
				})},
			},
			want: []yasmclient.YasmMetrics{{
				Tags: Tags{"ctype": "prod", "unit": "juggler-agent-rtc", "unit_type": "porto-daemon"},
				TTL:  signalTTL,
				Values: []yasmclient.YasmValue{
					{
						Name:  "unit_ready_thhh",
						Value: 1,
					}, {
						Name:  "unit_pending_thhh",
						Value: 0,
					}, {
						Name:  "unit_changed_thhh",
						Value: 0,
					}, {
						Name:  "unit_running_thhh",
						Value: 1,
					}, {
						Name:  "unit_restart_count_thhh",
						Value: 0,
					}, {
						Name:  "unit_respawn_count_thhh",
						Value: 0,
					}},
			}},
		}, {
			name: "PackageSet",
			args: args{
				tags: Tags{"ctype": "prod"},
				slots: map[string]slot.Slot{"upstream-packages": slot.NewSlot(&pb.Slot{
					Name: "upstream-packages",
					Revs: []*pb.Rev{{
						Target: pb.RevisionTarget_CURRENT,
						Meta: &pb.RevisionMeta{
							Kind: "PackageSet",
						},
					}},
					Status: &pb.SlotStatus{
						Pending:      falseCond(),
						Ready:        trueCond(),
						Removed:      falseCond(),
						Throttled:    falseCond(),
						Changed:      trueCond(),
						Conflicted:   falseCond(),
						Installed:    trueCond(),
						Running:      trueCond(),
						RestartCount: 0,
					},
					Meta: &pb.SlotMeta{},
				})},
			},
			want: []yasmclient.YasmMetrics{{
				Tags: Tags{"ctype": "prod", "unit": "upstream-packages", "unit_type": "package-set"},
				TTL:  signalTTL,
				Values: []yasmclient.YasmValue{{
					Name:  "unit_ready_thhh",
					Value: 1,
				}, {
					Name:  "unit_pending_thhh",
					Value: 0,
				}, {
					Name:  "unit_changed_thhh",
					Value: 1,
				}},
			}},
		}, {
			name: "removed",
			args: args{
				tags: Tags{"ctype": "prod"},
				slots: map[string]slot.Slot{"upstream-packages": slot.NewSlot(&pb.Slot{
					Name: "upstream-packages",
					Revs: []*pb.Rev{{
						Target: pb.RevisionTarget_CURRENT,
						Meta: &pb.RevisionMeta{
							Kind: "PackageSet",
						},
					}},
					Status: &pb.SlotStatus{
						Pending:    falseCond(),
						Ready:      trueCond(),
						Removed:    trueCond(),
						Throttled:  falseCond(),
						Changed:    trueCond(),
						Conflicted: falseCond(),
						Installed:  trueCond(),
						Running:    trueCond(),
					},
					Meta: &pb.SlotMeta{},
				})},
			},
			want: []yasmclient.YasmMetrics{{
				Tags: Tags{"ctype": "prod", "unit": "upstream-packages", "unit_type": "package-set"},
				TTL:  signalTTL,
				Values: []yasmclient.YasmValue{
					{
						Name:  "unit_ready_thhh",
						Value: 2,
					}, {
						Name:  "unit_pending_thhh",
						Value: 0,
					}, {
						Name:  "unit_changed_thhh",
						Value: 1,
					}},
			}},
		}, {
			name: "PortoDaemon after reboot",
			args: args{
				tags: Tags{"ctype": "prod"},
				slots: map[string]slot.Slot{"juggler-agent-rtc": slot.NewSlot(&pb.Slot{
					Name: "juggler-agent-rtc",
					Revs: []*pb.Rev{{
						Target: pb.RevisionTarget_CURRENT,
						Meta: &pb.RevisionMeta{
							Kind: "PortoDaemon",
						},
					}},
					Status: &pb.SlotStatus{
						Pending:      falseCond(),
						Ready:        trueCond(),
						Removed:      falseCond(),
						Throttled:    falseCond(),
						Changed:      trueCond(),
						Conflicted:   falseCond(),
						Installed:    trueCond(),
						Running:      trueCond(),
						RestartCount: 0,
						RespawnCount: 0,
					},
					Meta: &pb.SlotMeta{},
				})},
				rebooted: true,
			},
			want: []yasmclient.YasmMetrics{{
				Tags: Tags{"ctype": "prod", "unit": "juggler-agent-rtc", "unit_type": "porto-daemon"},
				TTL:  signalTTL,
				Values: []yasmclient.YasmValue{
					{
						Name:  "unit_ready_thhh",
						Value: 1,
					}, {
						Name:  "unit_pending_thhh",
						Value: 0,
					}, {
						Name:  "unit_changed_thhh",
						Value: 0,
					}, {
						Name:  "unit_running_thhh",
						Value: 1,
					}, {
						Name:  "unit_restart_count_thhh",
						Value: 0,
					}, {
						Name:  "unit_respawn_count_thhh",
						Value: 0,
					}},
			}},
		}, {
			name: "two units",
			args: args{
				tags: Tags{"ctype": "prod"},
				slots: map[string]slot.Slot{"juggler-agent-rtc": slot.NewSlot(&pb.Slot{
					Name: "juggler-agent-rtc",
					Revs: []*pb.Rev{{
						Target: pb.RevisionTarget_CURRENT,
						Meta: &pb.RevisionMeta{
							Kind: "PortoDaemon",
						},
					}},
					Status: &pb.SlotStatus{
						Pending:      falseCond(),
						Ready:        trueCond(),
						Removed:      falseCond(),
						Throttled:    falseCond(),
						Changed:      falseCond(),
						Conflicted:   falseCond(),
						Installed:    trueCond(),
						Running:      trueCond(),
						RestartCount: 0,
						RespawnCount: 0,
					},
					Meta: &pb.SlotMeta{},
				}), "upstream-packages": slot.NewSlot(&pb.Slot{
					Name: "upstream-packages",
					Revs: []*pb.Rev{{
						Target: pb.RevisionTarget_CURRENT,
						Meta: &pb.RevisionMeta{
							Kind: "PackageSet",
						},
					}},
					Status: &pb.SlotStatus{
						Pending:      falseCond(),
						Ready:        trueCond(),
						Removed:      falseCond(),
						Throttled:    falseCond(),
						Changed:      trueCond(),
						Conflicted:   falseCond(),
						Installed:    trueCond(),
						Running:      trueCond(),
						RestartCount: 0,
					},
					Meta: &pb.SlotMeta{},
				})},
			},
			want: []yasmclient.YasmMetrics{{
				Tags: Tags{"ctype": "prod", "unit": "juggler-agent-rtc", "unit_type": "porto-daemon"},
				TTL:  signalTTL,
				Values: []yasmclient.YasmValue{
					{
						Name:  "unit_ready_thhh",
						Value: 1,
					}, {
						Name:  "unit_pending_thhh",
						Value: 0,
					}, {
						Name:  "unit_changed_thhh",
						Value: 0,
					}, {
						Name:  "unit_running_thhh",
						Value: 1,
					}, {
						Name:  "unit_restart_count_thhh",
						Value: 0,
					}, {
						Name:  "unit_respawn_count_thhh",
						Value: 0,
					}},
			}, {
				Tags: Tags{"ctype": "prod", "unit": "upstream-packages", "unit_type": "package-set"},
				TTL:  signalTTL,
				Values: []yasmclient.YasmValue{{
					Name:  "unit_ready_thhh",
					Value: 1,
				}, {
					Name:  "unit_pending_thhh",
					Value: 0,
				}, {
					Name:  "unit_changed_thhh",
					Value: 1,
				}},
			}},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			yasm := ExtractYASM(tt.args.tags, tt.args.slots, tt.args.rebooted)
			if len(yasm) != len(tt.want) {
				t.Errorf("ExtractMonitoring() got = %v, want %v", yasm, tt.want)
			} else {
				for i := 0; i < len(yasm); i++ {
					assert.Equal(t, yasm[i], tt.want[i], fmt.Sprintf("ExtractMonitoring() got = %v, want %v", yasm, tt.want))
				}
			}
		})
	}
}

func TestNewYasmVisitorReuse(t *testing.T) {
	v := NewYasmVisitor(false)
	assert.Equal(t, make([]yasmclient.YasmValue, 0), v.Values())
	v.VisitUnit((*slot.Status)(&pb.SlotStatus{
		Pending:    trueCond(),
		Ready:      trueCond(),
		Removed:    falseCond(),
		Throttled:  trueCond(),
		Changed:    trueCond(),
		Conflicted: trueCond(),
		Installed:  trueCond(),
		Running:    trueCond(),
	}))
	assert.Equal(t, []yasmclient.YasmValue{
		{Name: "unit_ready_thhh", Value: 1},
		{Name: "unit_pending_thhh", Value: 1},
		{Name: "unit_changed_thhh", Value: 1},
	}, v.Values())
	// Assert empty after reset
	v.Reset()
	assert.Equal(t, make([]yasmclient.YasmValue, 0), v.Values())
	v.VisitUnit((*slot.Status)(&pb.SlotStatus{
		Pending:    falseCond(),
		Ready:      falseCond(),
		Removed:    falseCond(),
		Throttled:  falseCond(),
		Changed:    falseCond(),
		Conflicted: falseCond(),
		Installed:  falseCond(),
		Running:    falseCond(),
	}))
	// Assert that visitor can be reused
	assert.Equal(t, []yasmclient.YasmValue{
		{Name: "unit_ready_thhh", Value: 0},
		{Name: "unit_pending_thhh", Value: 0},
		{Name: "unit_changed_thhh", Value: 0},
	}, v.Values())
}

func trueCond() *pb.Condition {
	return &pb.Condition{
		Status:         "True",
		Message:        "True",
		TransitionTime: ptypes.TimestampNow(),
	}
}

func falseCond() *pb.Condition {
	return &pb.Condition{
		Status:         "False",
		Message:        "False",
		TransitionTime: ptypes.TimestampNow(),
	}
}
