package revjob

import (
	"fmt"

	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/units/specutil"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks"
	pb "a.yandex-team.ru/infra/hostctl/proto"
)

func NewCurrent(name, revID string, order ...tasks.Task) *RevisionJob {
	return &RevisionJob{
		ExecutionOrder: order,
		name:           name,
		revID:          revID,
		description:    "current",
	}
}

func CurrentFromRevision(name string, rev *slot.Rev, slotStatus *slot.Status) *RevisionJob {
	switch s := rev.Proto().Spec.(type) {
	case *pb.Rev_PackageSet:
		return CurrentFromPackageSet(name, rev.ID(), s.PackageSet, slotStatus)
	case *pb.Rev_PortoDaemon:
		return CurrentFromPortoDaemon(name, rev.ID(), s.PortoDaemon, slotStatus)
	case *pb.Rev_SystemService:
		return CurrentFromSystemService(name, rev.ID(), s.SystemService, slotStatus)
	case *pb.Rev_TimerJob:
		return CurrentFromTimerJob(name, rev.ID(), s.TimerJob, slotStatus)
	case *pb.Rev_Pod:
		return CurrentFromPod(name, rev.ID(), s.Pod, slotStatus)
	}
	panic(fmt.Sprintf("unknown revision spec kind: %s", rev.Proto().Spec))
}

func hostPodSetupTasks(revID string, spec *pb.HostPodSpec, slotStatus *slot.Status) []tasks.Task {
	installStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Installed)
	setupTasks := []tasks.Task{
		tasks.NewInstall(spec.Packages, installStatus),
		tasks.NewManage(spec.Files, installStatus),
	}
	runStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Running)
	for _, s := range spec.Services {
		setupTasks = append(setupTasks, systemServiceSetupTasks(s.Name, revID, s.Template, s.UpdatePolicy, runStatus)...)
	}
	for _, t := range spec.Timers {
		setupTasks = append(setupTasks, timerJobSetupTasks(t.Name, revID, t.Template, runStatus)...)
	}
	for _, d := range spec.PortoDaemons {
		setupTasks = append(setupTasks, tasks.NewRunPorto(d.Name, revID, d.Properties, runStatus, slotStatus))
	}
	return setupTasks
}

func CurrentFromPod(name, revID string, spec *pb.HostPodSpec, slotStatus *slot.Status) *RevisionJob {
	return NewCurrent(name, revID, hostPodSetupTasks(revID, spec, slotStatus)...)
}

func CurrentFromPackageSet(name, revID string, spec *pb.PackageSetSpec, slotStatus *slot.Status) *RevisionJob {
	installStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Installed)
	return NewCurrent(name, revID,
		tasks.NewInstall(spec.Packages, installStatus),
		tasks.NewManage(spec.Files, installStatus),
	)
}

func CurrentFromPortoDaemon(name, revID string, spec *pb.PortoDaemon, slotStatus *slot.Status) *RevisionJob {
	installStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Installed)
	return NewCurrent(name, revID,
		tasks.NewInstall(spec.Packages, installStatus),
		tasks.NewManage(spec.Files, installStatus),
		tasks.NewRunPorto(name, revID, spec.Properties, tasks.NewSimpleCond(slotStatus.Running), slotStatus),
	)
}
func systemServiceTemplateSetupTasks(name, revID string, template *pb.SystemdTemplate, updatePolicy *pb.UpdatePolicy,
	runStatus tasks.Condition, retryPolicy specutil.RetryPolicy) []tasks.Task {

	setupTasks := make([]tasks.Task, 0, len(template.Instances))
	for _, instance := range template.Instances {
		setupTasks = append(setupTasks, runSystemServiceTask(updatePolicy)(name+instance, revID, runStatus, retryPolicy))
	}
	return setupTasks
}

func systemServiceSetupTasks(name, revID string, template *pb.SystemdTemplate, updatePolicy *pb.UpdatePolicy,
	runStatus tasks.Condition) []tasks.Task {

	retryPolicy := specutil.RetryPolicyFromUpdatePolicyOrDefaults(updatePolicy)
	if sdTemplateOk(template) {
		return systemServiceTemplateSetupTasks(name, revID, template, updatePolicy, runStatus, retryPolicy)
	} else {
		return []tasks.Task{runSystemServiceTask(updatePolicy)(name, revID, runStatus, retryPolicy)}
	}
}

func CurrentFromSystemService(name, revID string, spec *pb.SystemServiceSpec, slotStatus *slot.Status) *RevisionJob {
	installStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Installed)
	runTasks := []tasks.Task{
		tasks.NewInstall(spec.Packages, installStatus),
		tasks.NewManage(spec.Files, installStatus),
	}
	runTasks = append(runTasks,
		systemServiceSetupTasks(name, revID, spec.Template, spec.UpdatePolicy, tasks.NewSetupFailureAccumulatorCond(slotStatus.Running))...,
	)
	return NewCurrent(name, revID, runTasks...)
}

func timerJobTemplateSetupTasks(name, revID string, template *pb.SystemdTemplate, runStatus tasks.Condition) []tasks.Task {
	setupTasks := make([]tasks.Task, 0, len(template.Instances))
	for _, instance := range template.Instances {
		setupTasks = append(setupTasks,
			tasks.NewRestartOneShotJob(name+instance, revID, runStatus),
			tasks.NewRunSystemdTimer(name+instance, revID, runStatus),
		)
	}
	return setupTasks
}

func timerJobSetupTasks(name, revID string, template *pb.SystemdTemplate, runStatus tasks.Condition) []tasks.Task {
	if sdTemplateOk(template) {
		return timerJobTemplateSetupTasks(name, revID, template, runStatus)
	} else {
		return []tasks.Task{tasks.NewRestartOneShotJob(name, revID, runStatus), tasks.NewRunSystemdTimer(name, revID, runStatus)}
	}
}

func CurrentFromTimerJob(name, revID string, spec *pb.TimerJobSpec, slotStatus *slot.Status) *RevisionJob {
	installStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Installed)
	runTasks := []tasks.Task{
		tasks.NewInstall(spec.Packages, installStatus),
		tasks.NewManage(spec.Files, installStatus),
	}
	runTasks = append(runTasks,
		timerJobSetupTasks(name, revID, spec.Template, tasks.NewSetupFailureAccumulatorCond(slotStatus.Running))...,
	)
	return NewCurrent(name, revID, runTasks...)
}

func runSystemServiceTask(p *pb.UpdatePolicy) tasks.NewRunSystemServiceTask {
	if p == nil {
		return tasks.NewRestartSystemService
	}
	switch p.Method {
	case "reload":
		return tasks.NewReloadSystemService
	case "restart":
		fallthrough
	default:
		return tasks.NewRestartSystemService
	}
}
