package revjob

import (
	"fmt"

	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks"
	pb "a.yandex-team.ru/infra/hostctl/proto"
)

func NewReinstall(name, revID string, order ...tasks.Task) *RevisionJob {
	return &RevisionJob{
		ExecutionOrder: order,
		name:           name,
		revID:          revID,
		description:    "reinstall",
	}
}

func NewReinstallFromRevAndSlot(name string, rev *slot.Rev, slot slot.Slot) *RevisionJob {
	slotStatus := slot.Status()
	switch s := rev.Proto().Spec.(type) {
	case *pb.Rev_PackageSet:
		return ReinstallFromPackageSet(name, rev.ID(), s.PackageSet, slotStatus)
	case *pb.Rev_PortoDaemon:
		return ReinstallFromPortoDaemon(name, rev.ID(), s.PortoDaemon, slotStatus)
	case *pb.Rev_SystemService:
		return ReinstallFromSystemService(name, rev.ID(), s.SystemService, slotStatus)
	case *pb.Rev_TimerJob:
		return ReinstallFromTimerJob(name, rev.ID(), s.TimerJob, slotStatus)
	case *pb.Rev_Pod:
		return ReinstallFromPod(name, rev.ID(), s.Pod, slotStatus)
	}
	panic(fmt.Sprintf("unknown revision spec kind: %s", rev.Proto().Spec))
}

func ReinstallFromPod(name, revID string, spec *pb.HostPodSpec, slotStatus *slot.Status) *RevisionJob {
	teardownTasks := hostPodTeardownTasks(spec, slotStatus)
	setupTasks := hostPodSetupTasks(revID, spec, slotStatus)
	return NewReinstall(name, revID, append(teardownTasks, setupTasks...)...)
}

func ReinstallFromPackageSet(name, revID string, spec *pb.PackageSetSpec, slotStatus *slot.Status) *RevisionJob {
	installStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Installed)
	removeStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Installed)
	return NewReinstall(name, revID,
		tasks.NewRemove(paths(spec.Files), removeStatus),
		tasks.NewUninstall(pkgNames(spec.Packages), removeStatus),
		tasks.NewInstall(spec.Packages, installStatus),
		tasks.NewManage(spec.Files, installStatus),
	)
}

func ReinstallFromPortoDaemon(name, revID string, spec *pb.PortoDaemon, slotStatus *slot.Status) *RevisionJob {
	installStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Installed)
	removeStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Installed)
	runStatus := tasks.NewSimpleCond(slotStatus.Running)
	return NewReinstall(name, revID,
		tasks.NewShutdownPorto(name, runStatus),
		tasks.NewRemove(paths(spec.Files), removeStatus),
		tasks.NewUninstall(pkgNames(spec.Packages), removeStatus),
		tasks.NewInstall(spec.Packages, installStatus),
		tasks.NewManage(spec.Files, installStatus),
		tasks.NewRunPorto(name, revID, spec.Properties, runStatus, slotStatus),
	)
}

func ReinstallFromSystemService(name, revID string, spec *pb.SystemServiceSpec, slotStatus *slot.Status) *RevisionJob {
	reinstallTasks := make([]tasks.Task, 0)
	installStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Installed)
	removeStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Installed)
	runStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Running)
	shutdownStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Running)
	reinstallTasks = append(reinstallTasks, systemServiceTeardownTasks(name, spec.Template, spec.UpdatePolicy, shutdownStatus)...)
	reinstallTasks = append(reinstallTasks,
		tasks.NewRemove(paths(spec.Files), removeStatus),
		tasks.NewUninstall(pkgNames(spec.Packages), removeStatus),
		tasks.NewInstall(spec.Packages, installStatus),
		tasks.NewManage(spec.Files, installStatus),
	)
	reinstallTasks = append(reinstallTasks, systemServiceSetupTasks(name, revID, spec.Template, spec.UpdatePolicy, runStatus)...)
	return NewReinstall(name, revID, reinstallTasks...)
}

func ReinstallFromTimerJob(name, revID string, spec *pb.TimerJobSpec, slotStatus *slot.Status) *RevisionJob {
	reinstallTasks := make([]tasks.Task, 0)
	installStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Installed)
	removeStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Installed)
	runStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Running)
	shutdownStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Running)
	reinstallTasks = append(reinstallTasks, timerJobTeardownTasks(name, spec.Template, shutdownStatus)...)
	reinstallTasks = append(reinstallTasks,
		tasks.NewRemove(paths(spec.Files), removeStatus),
		tasks.NewUninstall(pkgNames(spec.Packages), removeStatus),
		tasks.NewInstall(spec.Packages, installStatus),
		tasks.NewManage(spec.Files, installStatus),
	)
	reinstallTasks = append(reinstallTasks, timerJobSetupTasks(name, revID, spec.Template, runStatus)...)
	return NewReinstall(name, revID, reinstallTasks...)
}
