package revjob

import (
	"fmt"

	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/units/specutil"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks"
	pb "a.yandex-team.ru/infra/hostctl/proto"
)

func NewRemoved(name, revID string, order ...tasks.Task) *RevisionJob {
	return &RevisionJob{
		ExecutionOrder: order,
		name:           name,
		revID:          revID,
		description:    "removed",
	}
}

func RemovedFromRevision(name string, rev *slot.Rev, slotStatus *slot.Status) *RevisionJob {
	switch s := rev.Proto().Spec.(type) {
	case *pb.Rev_PackageSet:
		return RemovedFromPackageSet(name, rev.ID(), s.PackageSet, slotStatus)
	case *pb.Rev_PortoDaemon:
		return RemovedFromPortoDaemon(name, rev.ID(), s.PortoDaemon, slotStatus)
	case *pb.Rev_SystemService:
		return RemovedFromSystemService(name, rev.ID(), s.SystemService, slotStatus)
	case *pb.Rev_TimerJob:
		return RemovedFromTimerJob(name, rev.ID(), s.TimerJob, slotStatus)
	case *pb.Rev_Pod:
		return RemovedFromHostPod(name, rev.ID(), s.Pod, slotStatus)
	}
	panic(fmt.Sprintf("unknown revision spec kind: %s", rev.Proto().Spec))
}

func hostPodTeardownTasks(spec *pb.HostPodSpec, slotStatus *slot.Status) []tasks.Task {
	teardownTasks := make([]tasks.Task, 0)
	shutdownStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Running)
	for _, d := range spec.PortoDaemons {
		teardownTasks = append(teardownTasks, tasks.NewShutdownPorto(d.Name, shutdownStatus))
	}
	for _, t := range spec.Timers {
		teardownTasks = append(teardownTasks, timerJobTeardownTasks(t.Name, t.Template, shutdownStatus)...)
	}
	for _, s := range spec.Services {
		teardownTasks = append(teardownTasks, systemServiceTeardownTasks(s.Name, s.Template, s.UpdatePolicy, shutdownStatus)...)
	}
	removeStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Installed)
	teardownTasks = append(teardownTasks,
		tasks.NewRemove(paths(spec.Files), removeStatus),
		tasks.NewUninstall(pkgNames(spec.Packages), removeStatus),
	)
	return teardownTasks
}

func RemovedFromHostPod(name, revID string, spec *pb.HostPodSpec, slotStatus *slot.Status) *RevisionJob {
	return NewRemoved(name, revID, hostPodTeardownTasks(spec, slotStatus)...)
}

func RemovedFromPackageSet(name, revID string, spec *pb.PackageSetSpec, slotStatus *slot.Status) *RevisionJob {
	removeStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Installed)
	return NewRemoved(name, revID,
		tasks.NewRemove(paths(spec.Files), removeStatus),
		tasks.NewUninstall(pkgNames(spec.Packages), removeStatus),
	)
}

func RemovedFromPortoDaemon(name, revID string, spec *pb.PortoDaemon, slotStatus *slot.Status) *RevisionJob {
	removeStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Installed)
	return NewRemoved(name, revID,
		tasks.NewShutdownPorto(name, tasks.NewSimpleCond(slotStatus.Running)),
		tasks.NewRemove(paths(spec.Files), removeStatus),
		tasks.NewUninstall(pkgNames(spec.Packages), removeStatus),
	)
}

func systemServiceTemplateTeardownTasks(name string, template *pb.SystemdTemplate, retryPolicy specutil.RetryPolicy,
	shutdownStatus tasks.Condition) []tasks.Task {

	teardownTasks := make([]tasks.Task, 0, len(template.Instances))
	for _, instance := range template.Instances {
		teardownTasks = append(teardownTasks, tasks.NewShutdownSystemService(name+instance, shutdownStatus, retryPolicy))
	}
	return teardownTasks
}

func systemServiceTeardownTasks(name string, template *pb.SystemdTemplate, updatePolicy *pb.UpdatePolicy,
	shutdownStatus tasks.Condition) []tasks.Task {

	retryPolicy := specutil.RetryPolicyFromUpdatePolicyOrDefaults(updatePolicy)
	if sdTemplateOk(template) {
		return systemServiceTemplateTeardownTasks(name, template, retryPolicy, shutdownStatus)
	} else {
		return []tasks.Task{
			tasks.NewShutdownSystemService(name, shutdownStatus, retryPolicy),
		}
	}
}

func RemovedFromSystemService(name, revID string, spec *pb.SystemServiceSpec, slotStatus *slot.Status) *RevisionJob {
	removeStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Installed)
	return NewRemoved(name, revID,
		append(systemServiceTeardownTasks(name, spec.Template, spec.UpdatePolicy,
			tasks.NewSetupFailureAccumulatorCond(slotStatus.Running)),
			tasks.NewRemove(paths(spec.Files), removeStatus),
			tasks.NewUninstall(pkgNames(spec.Packages), removeStatus),
		)...,
	)
}

func timerJobTemplateTeardownTasks(name string, template *pb.SystemdTemplate, runStatus tasks.Condition,
	retryPolicy specutil.RetryPolicy) []tasks.Task {

	teardownTasks := make([]tasks.Task, 0, len(template.Instances)*2)
	for _, instance := range template.Instances {
		teardownTasks = append(teardownTasks,
			tasks.NewShutdownSystemTimer(name+instance, runStatus),
			tasks.NewShutdownSystemService(name+instance, runStatus, retryPolicy),
		)
	}
	return teardownTasks
}

func timerJobTeardownTasks(name string, template *pb.SystemdTemplate, runStatus tasks.Condition) []tasks.Task {
	retryPolicy := specutil.RetryPolicyFromUpdatePolicyOrDefaults(nil)
	if sdTemplateOk(template) {
		return timerJobTemplateTeardownTasks(name, template, runStatus, retryPolicy)
	} else {
		return []tasks.Task{
			tasks.NewShutdownSystemTimer(name, runStatus),
			tasks.NewShutdownSystemService(name, runStatus, retryPolicy),
		}
	}
}

func RemovedFromTimerJob(name, revID string, spec *pb.TimerJobSpec, slotStatus *slot.Status) *RevisionJob {
	removeStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Installed)
	return NewRemoved(name, revID,
		append(
			timerJobTeardownTasks(name, spec.Template, tasks.NewSetupFailureAccumulatorCond(slotStatus.Running)),
			tasks.NewRemove(paths(spec.Files), removeStatus),
			tasks.NewUninstall(pkgNames(spec.Packages), removeStatus),
		)...,
	)
}
