package revjob

import (
	"fmt"

	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/units/specutil"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks"
	pb "a.yandex-team.ru/infra/hostctl/proto"
)

func NewRestart(name, revID string, order ...tasks.Task) *RevisionJob {
	return &RevisionJob{
		ExecutionOrder: order,
		name:           name,
		revID:          revID,
		description:    "restart",
	}
}

func NewRestartFromSlotAndRev(name string, rev *slot.Rev, slot slot.Slot) *RevisionJob {
	slotStatus := slot.Status()
	switch s := rev.Proto().Spec.(type) {
	case *pb.Rev_PortoDaemon:
		return RestartFromPortoDaemon(name, rev.ID(), s.PortoDaemon, slotStatus)
	case *pb.Rev_SystemService:
		return RestartFromSystemService(name, rev.ID(), s.SystemService, slotStatus)
	case *pb.Rev_TimerJob:
		return RestartFromTimerJob(name, rev.ID(), s.TimerJob, slotStatus)
	case *pb.Rev_Pod:
		return RestartFromPod(name, rev.ID(), s.Pod, slotStatus)
	}
	panic(fmt.Sprintf("unknown revision spec kind: %s", rev.Proto().Spec))
}

func RestartFromPod(name, revID string, spec *pb.HostPodSpec, slotStatus *slot.Status) *RevisionJob {
	runStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Running)
	shutdownStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Running)
	// Restart tasks order differs from reinstall
	// as we assume that it is safe to restart each entity in any order
	cycleTasks := make([]tasks.Task, 0)
	for _, s := range spec.Services {
		cycleTasks = append(cycleTasks, systemServiceCycleTasks(s.Name, revID, s.Template, s.UpdatePolicy, runStatus, shutdownStatus)...)
	}
	for _, t := range spec.Timers {
		cycleTasks = append(cycleTasks, timerJobCycleTasks(t.Name, revID, t.Template, runStatus, shutdownStatus)...)
	}
	for _, d := range spec.PortoDaemons {
		cycleTasks = append(cycleTasks,
			tasks.NewShutdownPorto(name, runStatus),
			tasks.NewRunPorto(d.Name, revID, d.Properties, runStatus, slotStatus),
		)
	}
	return NewRestart(name, revID, cycleTasks...)
}

func RestartFromPortoDaemon(name, revID string, spec *pb.PortoDaemon, slotStatus *slot.Status) *RevisionJob {
	runStatus := tasks.NewSimpleCond(slotStatus.Running)
	return NewRestart(name, revID,
		tasks.NewShutdownPorto(name, runStatus),
		tasks.NewRunPorto(name, revID, spec.Properties, runStatus, slotStatus),
	)
}

func systemServiceTemplateCycleTasks(name, revID string, template *pb.SystemdTemplate, retryPolicy specutil.RetryPolicy,
	runStatus, shutdownStatus tasks.Condition) []tasks.Task {

	reinstallTasks := make([]tasks.Task, 0, len(template.Instances)*2)
	for _, instance := range template.Instances {
		reinstallTasks = append(reinstallTasks, tasks.NewShutdownSystemService(name+instance, shutdownStatus, retryPolicy))
	}
	for _, instance := range template.Instances {
		reinstallTasks = append(reinstallTasks, tasks.NewRestartSystemService(name+instance, revID, runStatus, retryPolicy))
	}
	return reinstallTasks
}

func systemServiceCycleTasks(name, revID string, template *pb.SystemdTemplate, updatePolicy *pb.UpdatePolicy,
	runStatus, shutdownStatus tasks.Condition) []tasks.Task {

	retryPolicy := specutil.RetryPolicyFromUpdatePolicyOrDefaults(updatePolicy)
	if sdTemplateOk(template) {
		return systemServiceTemplateCycleTasks(name, revID, template, retryPolicy, runStatus, shutdownStatus)
	} else {
		return []tasks.Task{
			tasks.NewShutdownSystemService(name, shutdownStatus, retryPolicy),
			tasks.NewRestartSystemService(name, revID, runStatus, retryPolicy),
		}
	}
}

func RestartFromSystemService(name, revID string, spec *pb.SystemServiceSpec, slotStatus *slot.Status) *RevisionJob {
	runStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Running)
	shutdownStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Running)
	return NewRestart(name, revID,
		systemServiceCycleTasks(name, revID, spec.Template, spec.UpdatePolicy, runStatus, shutdownStatus)...,
	)
}

func timerJobTemplateCycleTasks(name, revID string, template *pb.SystemdTemplate,
	runStatus, shutdownStatus tasks.Condition, retryPolicy specutil.RetryPolicy) []tasks.Task {

	reinstallTasks := make([]tasks.Task, 0, len(template.Instances)*4)
	for _, instance := range template.Instances {
		reinstallTasks = append(reinstallTasks,
			tasks.NewShutdownSystemTimer(name+instance, shutdownStatus),
			tasks.NewShutdownSystemService(name+instance, shutdownStatus, retryPolicy),
		)
	}
	for _, instance := range template.Instances {
		reinstallTasks = append(reinstallTasks,
			tasks.NewRestartOneShotJob(name+instance, revID, runStatus),
			tasks.NewRunSystemdTimer(name+instance, revID, runStatus),
		)
	}
	return reinstallTasks
}

func timerJobCycleTasks(name, revID string, template *pb.SystemdTemplate, runStatus, shutdownStatus tasks.Condition) []tasks.Task {
	retryPolicy := specutil.RetryPolicyFromUpdatePolicyOrDefaults(nil)
	if sdTemplateOk(template) {
		return timerJobTemplateCycleTasks(name, revID, template, runStatus, shutdownStatus, retryPolicy)
	} else {
		return []tasks.Task{
			tasks.NewShutdownSystemTimer(name, runStatus),
			tasks.NewShutdownSystemService(name, runStatus, retryPolicy),
			tasks.NewRestartOneShotJob(name, revID, runStatus),
			tasks.NewRunSystemdTimer(name, revID, runStatus),
		}
	}
}

func RestartFromTimerJob(name, revID string, spec *pb.TimerJobSpec, slotStatus *slot.Status) *RevisionJob {
	runStatus := tasks.NewSetupFailureAccumulatorCond(slotStatus.Running)
	shutdownStatus := tasks.NewTeardownFailureAccumulatorCond(slotStatus.Running)
	return NewRestart(name, revID,
		timerJobCycleTasks(name, revID, spec.Template, runStatus, shutdownStatus)...,
	)
}
