package revjob

import (
	"fmt"

	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks"
	pb "a.yandex-team.ru/infra/hostctl/proto"
)

// hostctl unit revision job interface.
type RevisionJob struct {
	ExecutionOrder []tasks.Task
	name           string
	revID          string
	description    string
}

// Converts slot to job set.
func FromSlot(slot slot.Slot) Set {
	// Slot can contain:
	//    * current revision (must be installed and run)
	//    * old revisions (to be removed)
	//   But! Old revisions and current can intersect:
	//   e.g old and current revision have same package installed
	//   and we do not want to remove package to install it again
	//   a moment later.
	//   Thus we want to "prune" old revisions remove steps from
	//   re-used packages and files.
	//   To build such set of jobs (can have several per revision) we:
	//    * build job for current revision (if any)
	//    * build and prune every job for every removed revision
	jobs := make(Set, 0)
	for _, rev := range slot.Removed() {
		rmJob := FromRev(rev, slot.Status(), slot.Name())
		jobs = append(jobs, rmJob)
	}
	// execute after all removed
	cur := slot.Current()
	if cur != nil {
		jobs = append(jobs, FromRev(cur, slot.Status(), slot.Name()))
	}
	return jobs
}

type JobStatus struct {
	ID          string
	Changes     *changelog.ChangeLog
	Description string
	Err         error
}

type Set []*RevisionJob

func (jobs Set) Execute(e *env.Env) []*JobStatus {
	statuses := make([]*JobStatus, 0)
	for _, j := range []*RevisionJob(jobs) {
		ch, err := j.Execute(e)
		s := &JobStatus{
			ID:          j.RevID(),
			Changes:     ch,
			Description: j.Description(),
			Err:         err,
		}
		statuses = append(statuses, s)
	}
	return statuses
}

func (jobs Set) Prune() {
	for i := 0; i < len(jobs); i++ {
		ja := jobs[i]
		for j := i + 1; j < len(jobs); j++ {
			jb := jobs[j]
			Prune(ja, jb)
		}
	}
}

func FromRev(rev *slot.Rev, status *slot.Status, name string) *RevisionJob {
	switch rev.Proto().Target {
	case pb.RevisionTarget_REMOVED:
		return RemovedFromRevision(name, rev, status)
	case pb.RevisionTarget_CURRENT:
		return CurrentFromRevision(name, rev, status)
	}
	panic(fmt.Sprintf("unknown revision target: %s", rev.Proto().Target))
}

func (j *RevisionJob) RevID() string {
	return j.revID
}

func (j *RevisionJob) Description() string {
	return j.description
}

func (j *RevisionJob) SlotName() string {
	return j.name
}

func (j *RevisionJob) Execute(env *env.Env) (*changelog.ChangeLog, error) {
	ch := changelog.New()
	for _, t := range j.ExecutionOrder {
		err := t.Execute(env, ch)
		if err != nil {
			env.L.Errorf("Failed to execute %s@%.11s (%s): %s", j.name, j.revID, j.description, err)
			return ch, err
		}
	}
	return ch, nil
}

// IterTasks all tasks of job
// Can be used when we want modify source list of tasks. E.g prune
func (j *RevisionJob) IterTasks(fun func(task tasks.Task)) {
	for _, t := range j.ExecutionOrder {
		fun(t)
	}
}

// FilterTask iter all tasks of job with func f().
// Replace tasks of source list with modified.
// If f() returns nil, skip task and trim len of source list
// Can be used when we want filter and modify source list of tasks. E.g skip removing tasks or prune tasks
func (j *RevisionJob) FilterTask(fun func(task tasks.Task) tasks.Task) {
	i2 := 0
	for i1 := 0; i1 < len(j.ExecutionOrder); i1++ {
		source := j.ExecutionOrder[i1]
		modified := fun(source)
		if modified != nil {
			j.ExecutionOrder[i2] = modified
			i2++
		}
	}
	j.ExecutionOrder = j.ExecutionOrder[:i2]
}

func Plan(j *RevisionJob) tasks.Plan {
	plan := make([]map[string]string, 0)
	for _, t := range j.ExecutionOrder {
		plan = t.Plan(plan)
	}
	return plan
}

func Prune(a, b *RevisionJob) {
	prune(a, b)
	prune(b, a)
}

// SkipRemovingTasks revision without described resources.
// Usage example:
//   * We want to remove file from spec, but left on host
//   * Describe file in annotation
//   * Remove file from spec
//   * Remove forget annotation
func SkipRemovingTasks(j *RevisionJob, skipRemove *pb.SkipRemovePhase) {
	j.FilterTask(func(t tasks.Task) tasks.Task {
		if uninstall, ok := t.(*tasks.PackageUninstall); ok {
			uninstall.Prune(skipRemove.Packages)
		}
		if remove, ok := t.(*tasks.FileRemove); ok {
			remove.Prune(skipRemove.Files)
		}
		if skipRemove.Daemon {
			switch t.(type) {
			case *tasks.SystemdShutdown:
				return nil
			case *tasks.PortoShutdown:
				return nil
			}
		}
		return t
	})
}

func prune(a, b *RevisionJob) {
	a.IterTasks(func(ta tasks.Task) {
		b.IterTasks(func(tb tasks.Task) {
			switch task := ta.(type) {
			case *tasks.FileManage:
				remove, ok := tb.(*tasks.FileRemove)
				if ok {
					remove.Prune(paths(task.Files()))
				}
			case *tasks.PackageInstall:
				uninstall, ok := tb.(*tasks.PackageUninstall)
				if ok {
					uninstall.Prune(pkgNames(task.Packages()))
				}
			case *tasks.PortoRun:
				shutdown, ok := tb.(*tasks.PortoShutdown)
				if ok {
					shutdown.Prune(task.Name())
				}
			case *tasks.SystemdRun:
				shutdown, ok := tb.(*tasks.SystemdShutdown)
				if ok {
					shutdown.Prune(task.Name(), task.Kind())
				}
			}
		})
	})
}

func paths(files []*pb.ManagedFile) []string {
	paths := make([]string, len(files))
	for i, f := range files {
		paths[i] = f.Path
	}
	return paths
}

func pkgNames(packages []*pb.SystemPackage) []string {
	names := make([]string, len(packages))
	for i, f := range packages {
		names[i] = f.Name
	}
	return names
}

func sdTemplateOk(t *pb.SystemdTemplate) bool {
	return t != nil && len(t.Instances) > 0
}
