package storage

import (
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/test/assertpb"
	"github.com/stretchr/testify/assert"
	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/known/timestamppb"
	"io/fs"
	"os"
	"testing"
	"testing/fstest"
	"time"
)

func Test_storage_Load(t *testing.T) {
	okState := &pb.HostctlState{
		Slots:      make([]*pb.Slot, 0),
		UnitsTs:    timestamppb.Now(),
		Monitoring: &pb.HostctlState_Monitoring{BootId: "4d35afa4-52e3-4d98-916d-173eaaa3e7b0"},
	}
	stateBytes := func(state *pb.HostctlState) []byte {
		b, _ := proto.Marshal(state)
		return b
	}
	tests := []struct {
		name string
		path string
		open func() (fs.File, error)
		want *pb.HostctlState
		err  string
	}{{
		name: "ok state",
		path: "state_file",
		open: func() (fs.File, error) {
			fsys := fstest.MapFS{"state": &fstest.MapFile{
				Data:    stateBytes(okState),
				Mode:    fs.ModePerm,
				ModTime: time.Now(),
			}}
			return fsys.Open("state")
		},
		want: okState,
	}, {
		name: "empty file",
		path: "state_file",
		open: func() (fs.File, error) {
			return nil, os.ErrNotExist
		},
		err: "failed open: file does not exist",
	}, {
		name: "dir",
		path: "state_file",
		open: func() (fs.File, error) {
			fsys := fstest.MapFS{"state": &fstest.MapFile{
				Data:    nil,
				Mode:    fs.ModeDir,
				ModTime: time.Now(),
			}}
			return fsys.Open("state")
		},
		err: "can not read state: state_file is dir",
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			s := &storage{
				path: tt.path,
				open: tt.open,
			}
			got, err := s.Load()
			if err != nil {
				assert.EqualError(t, err, tt.err)
				return
			} else if tt.err != "" {
				t.Error(err)
				return
			}
			assertpb.Equal(t, got, tt.want)
			assertpb.Equal(t, got, s.orig)
			if got == s.orig {
				t.Error("state.orig should not be pointer equal to state")
			}
		})
	}
}
