package tasks

import (
	"strings"

	"a.yandex-team.ru/infra/hostctl/pkg/pbutil"
	pb "a.yandex-team.ru/infra/hostctl/proto"
)

type Condition interface {
	Unknown(string)
	False(string)
	True(string)
}

type SimpleCondition pb.Condition

// NewSimpleCond Wraps *pb.Condition to match Condition interface.
func NewSimpleCond(cond *pb.Condition) *SimpleCondition {
	return (*SimpleCondition)(cond)
}

func (c *SimpleCondition) Unknown(msg string) {
	pbutil.UnknownCond((*pb.Condition)(c), msg)
}

func (c *SimpleCondition) False(msg string) {
	pbutil.FalseCond((*pb.Condition)(c), msg)
}

func (c *SimpleCondition) True(msg string) {
	pbutil.TrueCond((*pb.Condition)(c), msg)
}

type SetupFailureAccumulatorCondition struct {
	msgs []string
	cond *pb.Condition
}

// NewSetupFailureAccumulatorCond creates Condition accumulating messages for all non-.True() calls.
// Any .False() or .Unknown() call sets underlying condition to False without a way setting it back to True.
func NewSetupFailureAccumulatorCond(cond *pb.Condition) *SetupFailureAccumulatorCondition {
	pbutil.TrueCond(cond, "OK")
	return &SetupFailureAccumulatorCondition{
		msgs: make([]string, 0),
		cond: cond,
	}
}

func (c *SetupFailureAccumulatorCondition) Unknown(msg string) {
	c.msgs = append(c.msgs, msg)
	// set underlying condition to False as we don't want to track if it was failure or undefined behaviour
	pbutil.FalseCond(c.cond, strings.Join(c.msgs, ", "))
}

func (c *SetupFailureAccumulatorCondition) False(msg string) {
	c.msgs = append(c.msgs, msg)
	pbutil.FalseCond(c.cond, strings.Join(c.msgs, ", "))
}

func (c *SetupFailureAccumulatorCondition) True(_ string) {
}

type TeardownFailureAccumulatorCondition struct {
	msgs []string
	cond *pb.Condition
}

// NewTeardownFailureAccumulatorCond creates Condition accumulating messages for all non-.False() calls.
// Any .True() or .Unknown() call sets underlying condition to True without a way setting it back to False.
func NewTeardownFailureAccumulatorCond(cond *pb.Condition) *TeardownFailureAccumulatorCondition {
	pbutil.FalseCond(cond, "OK")
	return &TeardownFailureAccumulatorCondition{
		msgs: make([]string, 0),
		cond: cond,
	}
}

func (c *TeardownFailureAccumulatorCondition) Unknown(msg string) {
	c.msgs = append(c.msgs, msg)
	// set underlying condition to True as we don't want to track if it was teardown failure or undefined behaviour
	pbutil.TrueCond(c.cond, strings.Join(c.msgs, ", "))
}

func (c *TeardownFailureAccumulatorCondition) True(msg string) {
	c.msgs = append(c.msgs, msg)
	pbutil.TrueCond(c.cond, strings.Join(c.msgs, ", "))
}

func (c *TeardownFailureAccumulatorCondition) False(_ string) {
}
