package tasks

import (
	"errors"
	"fmt"
	"strings"

	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/env/pacman"

	pb "a.yandex-team.ru/infra/hostctl/proto"
)

func CheckConflicts(e *env.Env, install []*PackageInstall, uninstall []*PackageUninstall, remove []*FileRemove, manage []*FileManage) error {
	errs := make([]string, 0)
	pkgsInstall := make([]*pb.SystemPackage, 0)
	for _, inst := range install {
		if inst == nil {
			continue
		}
		pkgsInstall = append(pkgsInstall, inst.packages...)
	}
	errs = append(errs, checkPackagesConflicts(e, pkgsInstall)...)
	filesManage := make([]*pb.ManagedFile, 0)
	for _, mng := range manage {
		if mng == nil {
			continue
		}
		filesManage = append(filesManage, mng.files...)
	}
	errs = append(errs, checkFilesConflicts(filesManage)...)
	if len(errs) > 0 {
		return errors.New(strings.Join(errs, ";"))
	}
	return nil
}

func checkFilesConflicts(files []*pb.ManagedFile) []string {
	conflicts := make([]string, 0)
	for i := 0; i < len(files); i++ {
		for j := i + 1; j < len(files); j++ {
			a := files[i]
			b := files[j]
			if conflict := CheckFilesConflicts(a, b); conflict != "" {
				conflicts = append(conflicts, conflict)
			}
		}
	}
	return conflicts
}

func checkPackagesConflicts(e *env.Env, packages []*pb.SystemPackage) []string {
	conflicts := make([]string, 0)
	for i := 0; i < len(packages); i++ {
		for j := i + 1; j < len(packages); j++ {
			a := packages[i]
			b := packages[j]
			if conflict := CheckPkgConflicts(a, b); conflict != "" {
				conflicts = append(conflicts, conflict)
			}
		}
	}
	// trying to install packages with --dry-run to check shallow conflicts
	ps := make([]pacman.Package, 0)
	for _, p := range packages {
		ps = append(ps, pacman.Package{
			Name:    p.Name,
			Version: p.Version,
		})
	}
	err := e.Pacman.InstallSetDryRun(ps)
	if err != nil {
		conflicts = append(conflicts, err.Error())
	}
	return conflicts
}

func CheckFilesConflicts(a, b *pb.ManagedFile) string {
	if a.Path != b.Path {
		return ""
	}
	if a.Mode != b.Mode {
		return fmt.Sprintf("different mode in same file '%s': '%s', '%s'", a.Path, a.Mode, b.Mode)
	}
	if a.Content != b.Content {
		return fmt.Sprintf("different contents in same file '%s': '%s', '%s'", a.Path, a.Content, b.Content)
	}
	if a.Group != b.Group {
		return fmt.Sprintf("different groups for same file '%s': '%s', '%s'", a.Path, a.Group, b.Group)
	}
	if a.User != b.User {
		return fmt.Sprintf("different users for same file '%s': '%s', '%s'", a.Path, a.User, b.User)
	}
	return ""
}

func CheckPkgConflicts(a, b *pb.SystemPackage) string {
	if a.Name != b.Name {
		return ""
	}
	if a.Version != b.Version {
		return fmt.Sprintf("different version in same pkg '%s': '%s', '%s'", a.Name, a.Version, b.Version)
	}
	return ""
}
