package tasks

import (
	"fmt"
	"strings"

	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/env/file"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/slices"
	"gopkg.in/yaml.v2"
)

func NewManage(files []*pb.ManagedFile, installed Condition) *FileManage {
	return &FileManage{files, installed}
}

func NewEmptyManage() *FileManage {
	return &FileManage{make([]*pb.ManagedFile, 0), NewSimpleCond(&pb.Condition{})}
}

type FileManage struct {
	files     []*pb.ManagedFile
	installed Condition
}

func (u *FileManage) Add(file *pb.ManagedFile) {
	u.files = append(u.files, file)
}

func (u *FileManage) Execute(e *env.Env, changelog *changelog.ChangeLog) error {
	for _, f := range u.files {
		a, err := e.FileBuilder(f.Path, "", "", "", "").FromFs()
		if err != nil {
			u.installed.False(err.Error())
			return err
		}
		b := file.FromPb(e.FileBuilder, f)
		chgs := file.Changes(a, b)
		if chgs == nil {
			continue
		}
		e.L.Infof("Managing '%s'...", b.Path())
		err = b.Manage()
		if err != nil {
			e.L.Errorf("Failed: %s", err.Error())
			u.installed.False(err.Error())
			return err
		}
		chgsStr, _ := yaml.Marshal(chgs)
		changelog.Add("file.managed", string(chgsStr))
	}
	u.installed.True("OK")
	return nil
}

func (u *FileManage) Prune(paths []string) {
	prunedPaths := make([]*pb.ManagedFile, 0)
	for _, f := range u.files {
		if !slices.ContainsString(paths, f.Path) {
			prunedPaths = append(prunedPaths, f)
		}
	}
	u.files = prunedPaths
}

func (u *FileManage) Files() []*pb.ManagedFile {
	return u.files
}

func (u *FileManage) Plan(plan Plan) Plan {
	if len(u.files) == 0 {
		return plan
	}
	paths := make([]string, len(u.files))
	for i, f := range u.files {
		paths[i] = f.Path
	}
	plan = append(plan, map[string]string{"file.manage": fmt.Sprintf("paths: {%s}", strings.Join(paths, ", "))})
	return plan
}

func (u *FileManage) Description() TaskKind {
	return KindFileManage
}
