package tasks

import (
	"fmt"
	"os"
	"strings"

	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/library/go/slices"
)

func NewRemove(paths []string, installed Condition) *FileRemove {
	return &FileRemove{paths, installed}
}

type FileRemove struct {
	paths     []string
	installed Condition
}

func (u *FileRemove) Execute(e *env.Env, changelog *changelog.ChangeLog) error {
	for _, p := range u.paths {
		f, err := e.FileBuilder(p, "", "", "", "").FromFs()
		if err != nil {
			u.installed.Unknown(err.Error())
			return err
		}
		e.L.Infof("Removing '%s'...", p)
		err = f.Remove()
		if err != nil {
			if os.IsNotExist(err) {
				// do not add record to changelog
				continue
			}
			u.installed.Unknown(err.Error())
			return err
		}
		e.L.Infof("Removed '%s'", p)
		changelog.Add("file.removed", p)
	}
	u.installed.False("removed")
	return nil
}

func (u *FileRemove) Prune(paths []string) {
	prunedPaths := make([]string, 0)
	for _, p := range u.paths {
		if !slices.ContainsString(paths, p) {
			prunedPaths = append(prunedPaths, p)
		}
	}
	u.paths = prunedPaths
}

func (u *FileRemove) Plan(plan Plan) Plan {
	if len(u.paths) == 0 {
		return plan
	}
	return append(plan, map[string]string{"file.removed": fmt.Sprintf("paths: {%s}", strings.Join(u.paths, ", "))})
}

func (u *FileRemove) Description() TaskKind {
	return KindFileRemove
}
