package helper

import (
	"errors"
	"time"

	"a.yandex-team.ru/infra/hostctl/internal/behaviortree"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
)

const retryLimit = 3

func WithRetry(retryable behaviortree.Node, errMessage string) behaviortree.Node {
	return WithRetryN(retryable, errMessage, retryLimit)
}

func WithRetryN(retryable behaviortree.Node, errMessage string, limit int) behaviortree.Node {
	return behaviortree.New(
		func(children []behaviortree.Node) (behaviortree.Status, error) {
			for i := 0; i < limit; i++ {
				s, err := retryable.Tick()
				if err != nil {
					return behaviortree.Failure, err
				}
				if s == behaviortree.Success {
					return behaviortree.Success, nil
				}
			}
			return behaviortree.Failure, errors.New(errMessage)
		},
	)
}

func BusyWaitN(waitable behaviortree.Node, timeout, interval time.Duration, e *env.Env) behaviortree.Node {
	return behaviortree.New(
		func(children []behaviortree.Node) (behaviortree.Status, error) {
			deadline := time.Now().Add(timeout)
			e.L.Infof("Starting busy wait...")
			for {
				s, err := waitable.Tick()
				if err != nil {
					return behaviortree.Failure, err
				}
				if s == behaviortree.Success {
					e.L.Infof("Busy check succeed")
					return behaviortree.Success, nil
				}
				if time.Now().After(deadline) {
					return behaviortree.Failure, err
				}
				// skip retry if env mode is not real
				if e.Mode != env.RealMode {
					return behaviortree.Failure, err
				}
				e.L.Infof("Busy check failed: %s remains to deadline. "+
					"Waiting %s before next check", time.Until(deadline).Round(1*time.Second), interval)
				e.Sleep(interval)
			}
		},
	)
}

const (
	busyWaitTimeout  = 30 * time.Second
	busyWaitInterval = time.Second
)

func BusyWait(waitable behaviortree.Node, e *env.Env) behaviortree.Node {
	return BusyWaitN(waitable, busyWaitTimeout, busyWaitInterval, e)
}

func BusyWaitTimeout(waitable behaviortree.Node, e *env.Env, timeout time.Duration) behaviortree.Node {
	return BusyWaitN(waitable, timeout, busyWaitInterval, e)
}
