package tasks

import (
	"fmt"
	"strings"

	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/units/env/pacman"

	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	pb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/library/go/slices"
)

func NewInstall(packages []*pb.SystemPackage, installed Condition) *PackageInstall {
	return &PackageInstall{packages, installed}
}

type PackageInstall struct {
	packages  []*pb.SystemPackage
	installed Condition
}

func (u *PackageInstall) Packages() []*pb.SystemPackage {
	return u.packages
}

func (u *PackageInstall) Execute(e *env.Env, changelog *changelog.ChangeLog) error {
	diffPackages := make([]pacman.Package, 0)
	txPackages := make([]pacman.Package, len(u.packages))
	// check if we have diff, but install all packages in one apt-get tx
	names := make([]string, len(u.packages))
	for i, p := range u.packages {
		names[i] = p.Name
	}
	statuses, err := e.Pacman.List(names)
	if err != nil {
		u.installed.False(err.Error())
		return err
	}
	for i, pkg := range u.packages {
		status := statuses[pkg.Name]
		if !status.Installed || status.Version != pkg.Version {
			diffPackages = append(diffPackages, pacman.Package{Name: pkg.Name, Version: pkg.Version})
		} else {
			e.L.Infof("Package %s=%s is already installed", pkg.Name, pkg.Version)
		}
		txPackages[i] = pacman.Package{Name: pkg.Name, Version: pkg.Version}
	}
	if len(diffPackages) == 0 {
		u.installed.True("OK")
		return nil
	}
	out := make([]string, 0, len(txPackages))
	for _, p := range txPackages {
		out = append(out, fmt.Sprintf("%s=%s", p.Name, p.Version))
	}
	e.L.Infof("Installing: %s...", strings.Join(out, ", "))
	installErr := e.Pacman.InstallSet(txPackages)
	if installErr == nil {
		for _, p := range diffPackages {
			changelog.Add("pkg.installed", fmt.Sprintf("%s=%s", p.Name, p.Version))
		}
		u.installed.True("OK")
		return nil
	}
	txPackagesNames := make([]string, len(txPackages))
	for i, p := range txPackages {
		txPackagesNames[i] = p.Name
	}
	statuses, err = e.Pacman.List(txPackagesNames)
	if err != nil {
		e.L.Errorf("Failed to list packages: %s", err.Error())
		u.installed.False(installErr.Error())
		return installErr
	}
	for _, p := range txPackages {
		s := statuses[p.Name]
		if s.Name == p.Name && s.Version == p.Version && s.Installed {
			changelog.Add("pkg.installed", fmt.Sprintf("%s=%s", p.Name, p.Version))
		}
	}
	u.installed.False(installErr.Error())
	return installErr
}

func (u *PackageInstall) Prune(pkgNames []string) {
	pruned := make([]*pb.SystemPackage, 0)
	for _, p := range u.packages {
		if !slices.ContainsString(pkgNames, p.Name) {
			pruned = append(pruned, p)
		}
	}
	u.packages = pruned
}

func (u *PackageInstall) Plan(plan Plan) Plan {
	if len(u.packages) == 0 {
		return plan
	}
	pkgsStrs := make([]string, len(u.packages))
	for i, p := range u.packages {
		pkgsStrs[i] = fmt.Sprintf("%s=%s", p.Name, p.Version)
	}
	plan = append(plan, map[string]string{"package.install": fmt.Sprintf("pkgs: {%s}", strings.Join(pkgsStrs, ", "))})
	return plan
}

func (u *PackageInstall) Description() TaskKind {
	return KindPackageInstall
}
