package tasks

import (
	"fmt"
	"strings"

	"a.yandex-team.ru/infra/hostctl/internal/changelog"

	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/library/go/slices"
)

func NewUninstall(names []string, installed Condition) *PackageUninstall {
	return &PackageUninstall{names, installed}
}

type PackageUninstall struct {
	names     []string
	installed Condition
}

func (u *PackageUninstall) Add(name string) {
	u.names = append(u.names, name)
}

func (u *PackageUninstall) Names() []string {
	return u.names
}

func (u *PackageUninstall) Execute(e *env.Env, changelog *changelog.ChangeLog) error {
	if len(u.names) == 0 {
		return nil
	}
	todo := make([]string, 0)
	pkgs, err := e.Pacman.List(u.names)
	if err != nil {
		u.installed.Unknown(err.Error())
		return err
	}
	for _, p := range pkgs {
		if p.Installed {
			todo = append(todo, p.Name)
		}
	}
	if len(todo) <= 0 {
		u.installed.False("removed")
		return nil
	}
	e.L.Infof("Removing: %s...", strings.Join(todo, ", "))
	err = e.Pacman.PurgeSet(todo)
	if err != nil {
		u.installed.Unknown(err.Error())
		return err
	}
	u.installed.False("removed")
	for _, name := range todo {
		changelog.Add("pkg.purged", name)
	}
	return err
}

func (u *PackageUninstall) Prune(pkgNames []string) {
	pruned := make([]string, 0)
	for _, p := range u.names {
		if !slices.ContainsString(pkgNames, p) {
			pruned = append(pruned, p)
		}
	}
	u.names = pruned
}

func (u *PackageUninstall) Plan(plan Plan) Plan {
	if len(u.names) == 0 {
		return plan
	}
	plan = append(plan, map[string]string{"package.removed": fmt.Sprintf("pkgs: {%s}", strings.Join(u.names, ", "))})
	return plan
}

func (u *PackageUninstall) Description() TaskKind {
	return KindPackageUninstall
}
