package tasks

import (
	"errors"
	"fmt"
	"strconv"

	"a.yandex-team.ru/infra/hostctl/internal/behaviortree"
	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/slot"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/env/porto"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks/portostates"
	"a.yandex-team.ru/infra/hostctl/internal/yamlutil"
	pb "a.yandex-team.ru/infra/hostctl/proto"
)

func NewRunPorto(name, etag string, props *pb.PortoProperties, running Condition, slotStatus *slot.Status) *PortoRun {
	return &PortoRun{
		name:       name,
		etag:       etag,
		props:      props,
		running:    running,
		slotStatus: slotStatus,
	}
}

type PortoRun struct {
	name       string
	etag       string
	props      *pb.PortoProperties
	running    Condition
	slotStatus *slot.Status
}

func portoRespawnCount(e *env.Env, name string) int {
	_, countStr, err := e.Porto.Get(name, "respawn_count")
	if err != nil {
		e.L.Warnf("Failed to get respawn_count: %s", err)
	}
	if countStr == "" {
		return 0
	}
	count, err := strconv.ParseUint(countStr, 10, 64)
	if err != nil {
		e.L.Warnf("Failed to get respawn_count: %s", err)
	}
	return int(count)
}

func updateRespawns(slotStatus *slot.Status, current int) {
	old := slotStatus.TotalRespawnCount
	diff := int32(current) - old
	// Container's counter is zero after reboot, this may lead to
	// negative value which inapropriate for uint64 declared in spec
	if diff >= 0 {
		slotStatus.RespawnCount = diff
	} else {
		slotStatus.RespawnCount = int32(current)
	}
	slotStatus.TotalRespawnCount = int32(current)
}

func (s *PortoRun) Execute(e *env.Env, changelog *changelog.ChangeLog) error {
	// skipping empty task
	if s.name == "" {
		return nil
	}
	updateRespawns(s.slotStatus, portoRespawnCount(e, s.name))
	c := &portostates.Container{
		Name:   s.name,
		RevID:  s.etag,
		Props:  s.props,
		Status: &porto.Status{},
	}
	status, err := portostates.RunContainer()(e, changelog, c).Tick()
	if err != nil {
		s.running.Unknown(err.Error())
		return err
	}
	if status == behaviortree.Failure {
		err := fmt.Sprintf("failed to run porto container: Running='%t' Actual='%t'", c.Status.Running, c.Status.Etag == s.etag)
		s.running.Unknown(err)
		return errors.New(err)
	}
	if c.Status.Running {
		s.running.True(fmt.Sprintf("Running: %t", c.Status.Running))
	} else {
		s.running.False(fmt.Sprintf("Running: %t", c.Status.Running))
	}
	return err
}

func (s *PortoRun) Prune(names []string) {

}

func (s *PortoRun) Name() string {
	return s.name
}

func (s *PortoRun) Plan(plan Plan) Plan {
	if s.name == "" {
		return plan
	}
	p, _ := yamlutil.ProtoToYaml(s.props)
	plan = append(plan, map[string]string{"porto.container": fmt.Sprintf("name: %s\n properties:\n %s", s.name, string(p))})
	return plan
}

func (s *PortoRun) Description() TaskKind {
	return KindPortoRun
}
