package tasks

import (
	"errors"
	"fmt"

	"a.yandex-team.ru/infra/hostctl/internal/behaviortree"
	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/env/porto"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks/portostates"
	pb "a.yandex-team.ru/infra/hostctl/proto"
)

func NewShutdownPorto(name string, running Condition) *PortoShutdown {
	return &PortoShutdown{
		name:    name,
		running: running,
	}
}

type PortoShutdown struct {
	name    string
	running Condition
}

type shutdownPDTaskState struct {
	e         *env.Env
	changelog *changelog.ChangeLog
}

type shutdownPortoStateFn func(task *PortoShutdown, state *shutdownPDTaskState) (shutdownPortoStateFn, error)

func (s *PortoShutdown) Execute(e *env.Env, changelog *changelog.ChangeLog) error {
	// skipping empty task
	if s.name == "" {
		return nil
	}
	c := &portostates.Container{
		Name:   s.name,
		RevID:  "",
		Props:  &pb.PortoProperties{},
		Status: &porto.Status{},
	}
	status, err := portostates.ShutdownContainer()(e, changelog, c).Tick()
	if err != nil {
		s.running.Unknown(err.Error())
		return err
	}
	if status == behaviortree.Failure {
		err := fmt.Sprintf("failed to shutdown porto container: Running='%t'", c.Status.Running)
		s.running.Unknown(err)
		return errors.New(err)
	}
	if c.Status.Running {
		s.running.True(fmt.Sprintf("Running: %t", c.Status.Running))
	} else {
		s.running.False(fmt.Sprintf("Running: %t", c.Status.Running))
	}
	return err
}

func (s *PortoShutdown) Prune(name string) {
	if s.name == name {
		s.name = ""
	}
}

func (s *PortoShutdown) Plan(plan Plan) Plan {
	if s.name == "" {
		return plan
	}
	return append(plan, map[string]string{"porto.destroyed": s.name})
}

func (s *PortoShutdown) Description() TaskKind {
	return KindPortoShutdown
}
