package portostates

import (
	"a.yandex-team.ru/infra/hostctl/internal/behaviortree"
	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/env/porto"
	pb "a.yandex-team.ru/infra/hostctl/proto"
)

type Container struct {
	Name   string
	RevID  string
	Props  *pb.PortoProperties
	Status *porto.Status
}

type NewPortoNode func(e *env.Env, ch *changelog.ChangeLog, c *Container) behaviortree.Node

type StatusNode func(e *env.Env, c *Container) behaviortree.Node

/*
                [->]
                  |
        v------------------v
[ContainerRunning] [ContainerActual]
*/
func ContainerUpToDate() NewPortoNode {
	return func(e *env.Env, ch *changelog.ChangeLog, c *Container) behaviortree.Node {
		return behaviortree.New(
			behaviortree.Sequence,
			ContainerRunning()(e, c),
			ContainerActual()(e, ch, c),
		)
	}
}

/*
                [->]
                  |
        v------------------v
[ContainerNotRunning] [ContainerNotExist]
*/
func ContainerShutdowned() StatusNode {
	return func(e *env.Env, c *Container) behaviortree.Node {
		return behaviortree.New(
			behaviortree.Sequence,
			ContainerNotRunning()(e, c),
			ContainerNotExist()(e, c),
		)
	}
}

func ContainerRunning() StatusNode {
	return func(e *env.Env, c *Container) behaviortree.Node {
		return behaviortree.New(func(_ []behaviortree.Node) (behaviortree.Status, error) {
			e.L.Infof("'%s' container running=%t", c.Name, c.Status.Running)
			if c.Status.Running {
				return behaviortree.Success, nil
			}
			return behaviortree.Failure, nil
		})
	}
}

func ContainerNotRunning() StatusNode {
	return func(e *env.Env, c *Container) behaviortree.Node {
		return behaviortree.New(func(_ []behaviortree.Node) (behaviortree.Status, error) {
			e.L.Infof("'%s' container not running=%t", c.Name, !c.Status.Running)
			if !c.Status.Running {
				return behaviortree.Success, nil
			}
			return behaviortree.Failure, nil
		})
	}
}

func ContainerNotExist() StatusNode {
	return func(e *env.Env, c *Container) behaviortree.Node {
		return behaviortree.New(func(_ []behaviortree.Node) (behaviortree.Status, error) {
			e.L.Infof("'%s' container not exist=%t", c.Name, !c.Status.Exist)
			if !c.Status.Exist {
				return behaviortree.Success, nil
			}
			return behaviortree.Failure, nil
		})
	}
}

func ContainerActual() NewPortoNode {
	return func(e *env.Env, ch *changelog.ChangeLog, c *Container) behaviortree.Node {
		return behaviortree.New(func(_ []behaviortree.Node) (behaviortree.Status, error) {
			e.L.Infof("'%s' container actual=%t", c.Name, c.Status.Etag == c.RevID)
			if c.Status.Etag == c.RevID {
				return behaviortree.Success, nil
			}
			return behaviortree.Failure, nil
		})
	}
}

func UpdateStatus() NewPortoNode {
	return func(e *env.Env, ch *changelog.ChangeLog, c *Container) behaviortree.Node {
		return behaviortree.New(func(_ []behaviortree.Node) (behaviortree.Status, error) {
			e.L.Infof("Updating status for '%s' container", c.Name)
			status, err := e.Porto.Status(c.Name)
			if err != nil {
				return behaviortree.Failure, err
			}
			c.Status = status
			return behaviortree.Success, nil
		})
	}
}
