package tasks

import (
	"fmt"
	"time"

	"a.yandex-team.ru/infra/hostctl/internal/behaviortree"
	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/systemd"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks/systemdstates"
)

func NewSystemdJobRun(name, revID string, running Condition, updateMethod func(time.Duration) systemdstates.NewSystemdNode) *SystemdJobRun {
	unit := systemd.ServiceUnit(name)
	return &SystemdJobRun{
		revID:        revID,
		running:      running,
		updateMethod: updateMethod,
		unit:         unit,
	}
}

func NewReloadOneShotJob(name, revID string, running Condition) *SystemdJobRun {
	return NewSystemdJobRun(name, revID, running, systemdstates.ReloadService)
}

func NewRestartOneShotJob(name, revID string, running Condition) *SystemdJobRun {
	return NewSystemdJobRun(name, revID, running, systemdstates.RestartService)
}

type SystemdJobRun struct {
	revID        string
	running      Condition
	updateMethod func(time.Duration) systemdstates.NewSystemdNode
	unit         *systemd.Unit
}

/*
for SystemService we need
* enable systemd service
* ActiveState = active
* reload daemon + restart if NeedReloadDaemon
* restart container if outdated
* fail if job execution time more than 10min
*/
func (s *SystemdJobRun) Execute(e *env.Env, changelog *changelog.ChangeLog) error {
	// skipping empty task
	if s.unit.Name == "" {
		return nil
	}
	st := &systemd.UnitStatus{}
	status, err := systemdstates.RunOneShotJob(s.updateMethod)(e, changelog, s.unit, s.revID, st).Tick()
	if err != nil {
		s.running.Unknown(err.Error())
		return err
	}
	if st.SubState.Running() {
		s.running.True(fmt.Sprintf("SubState: %s", st.SubState))
	} else {
		s.running.False(fmt.Sprintf("SubState: %s", st.SubState))
	}
	if status == behaviortree.Failure {
		return fmt.Errorf("failed to run systemd unit: Exited='%t' Outdated='%t' ExecMainStatus='%s' Duration='%s'",
			!st.JobRunning(), st.Outdated, st.ExecMain.ExecMainStatus, st.Duration())
	}
	return nil
}

func (s *SystemdJobRun) Prune(name string) {
}

func (s *SystemdJobRun) Name() string {
	return s.unit.Name
}

func (s *SystemdJobRun) Plan(plan Plan) Plan {
	if s.unit.Name == "" {
		return plan
	}
	plan = append(plan, map[string]string{"systemd.daemon-reload": s.unit.FullName()})
	plan = append(plan, map[string]string{"systemd.start[oneshot]": s.unit.FullName()})
	plan = append(plan, map[string]string{"systemd.enable": s.unit.FullName()})
	return plan
}

func (s *SystemdJobRun) Description() TaskKind {
	return KindSystemdRun
}
