package tasks

import (
	"fmt"

	"a.yandex-team.ru/infra/hostctl/internal/behaviortree"
	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/systemd"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/specutil"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks/systemdstates"
)

type NewRunSystemServiceTask func(name, revID string, running Condition, retryPolicy specutil.RetryPolicy) *SystemdRun

func NewSystemdRun(name, revID string, running Condition, updater systemdstates.SystemdServiceUpdater, retryPolicy specutil.RetryPolicy) *SystemdRun {
	unit := systemd.ServiceUnit(name)
	return &SystemdRun{
		revID:       revID,
		running:     running,
		updater:     updater,
		unit:        unit,
		retryPolicy: retryPolicy,
	}
}

func NewReloadSystemService(name, revID string, running Condition, retryPolicy specutil.RetryPolicy) *SystemdRun {
	return NewSystemdRun(name, revID, running, systemdstates.SystemdServiceUpdater{
		UpdateMethod:      systemdstates.ReloadService,
		ActiveCheckMethod: systemdstates.ServiceActiveAfterReload,
	}, retryPolicy)
}

func NewRestartSystemService(name, revID string, running Condition, retryPolicy specutil.RetryPolicy) *SystemdRun {
	return NewSystemdRun(name, revID, running, systemdstates.SystemdServiceUpdater{
		UpdateMethod:      systemdstates.RestartService,
		ActiveCheckMethod: systemdstates.ServiceActive,
	}, retryPolicy)
}

type SystemdRun struct {
	revID       string
	running     Condition
	updater     systemdstates.SystemdServiceUpdater
	unit        *systemd.Unit
	retryPolicy specutil.RetryPolicy
}

/*
for SystemService we need
* enable systemd service
* ActiveState = active
* reload daemon + restart if NeedReloadDaemon
* restart container if outdated
*/
func (s *SystemdRun) Execute(e *env.Env, changelog *changelog.ChangeLog) error {
	// skipping empty task
	if s.unit.Name == "" {
		return nil
	}
	st := &systemd.UnitStatus{}
	status, err := systemdstates.RunSystemdService(s.retryPolicy, s.updater)(e, changelog, s.unit, s.revID, st).Tick()
	if st.SubState.Running() {
		s.running.True(fmt.Sprintf("SubState: %s", st.SubState))
	} else {
		s.running.False(fmt.Sprintf("SubState: %s", st.SubState))
	}
	if err != nil {
		return err
	}
	if status == behaviortree.Failure {
		return fmt.Errorf("failed to run systemd unit: ActiveState='%s' UnitFileState='%s'", st.ActiveState, st.UnitFileState)
	}
	return nil
}

func (s *SystemdRun) Prune(name string) {
}

func (s *SystemdRun) Name() string {
	return s.unit.Name
}

func (s *SystemdRun) Kind() systemd.UnitKind {
	return s.unit.Kind
}

func (s *SystemdRun) Plan(plan Plan) Plan {
	if s.unit.Name == "" {
		return plan
	}
	plan = append(plan, map[string]string{"systemd.daemon-reload": s.unit.FullName()})
	plan = append(plan, map[string]string{"systemd.start": s.unit.FullName()})
	plan = append(plan, map[string]string{"systemd.enable": s.unit.FullName()})
	return plan
}

func (s *SystemdRun) Description() TaskKind {
	return KindSystemdRun
}
