package tasks

import (
	"fmt"
	"time"

	"a.yandex-team.ru/infra/hostctl/internal/behaviortree"
	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/systemd"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/specutil"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks/systemdstates"
)

func NewShutdownSystemService(name string, running Condition, policy specutil.RetryPolicy) *SystemdShutdown {
	return &SystemdShutdown{
		unit:    systemd.ServiceUnit(name),
		running: running,
		timeout: policy.Timeout,
	}
}

func NewShutdownSystemTimer(name string, running Condition) *SystemdShutdown {
	return &SystemdShutdown{
		unit:    systemd.TimerUnit(name),
		running: running,
		// Use defaults for retry policy
		timeout: specutil.RetryPolicyFromUpdatePolicyOrDefaults(nil).Timeout,
	}
}

type SystemdShutdown struct {
	unit    *systemd.Unit
	running Condition
	timeout time.Duration
}

/*
For removing SystemService we need:
* disabled systemd service
* inactive systemd service

For this actions we have state machine
states:
1. checkSystemdInactiveDisabled
  > we getting status of systemd service
  > go to state 2
2. disableSystemdService
  > if UnitFileState != 'disabled'
  >   we need to disable service
  > go to state 3
3. stopSystemdService
  > if units property ActiveState != 'inactive'
  >   we need to stop service
  > go to state 1 to check status again
*/
func (s *SystemdShutdown) Execute(e *env.Env, changelog *changelog.ChangeLog) error {
	// skipping empty task
	if s.unit.Name == "" {
		return nil
	}
	st := &systemd.UnitStatus{}
	status, err := systemdstates.ShutdownSystemdService(s.timeout)(e, changelog, s.unit, "", st).Tick()
	if st.SubState.Running() {
		s.running.True(fmt.Sprintf("SubState: %s", st.SubState))
	} else {
		s.running.False(fmt.Sprintf("SubState: %s", st.SubState))
	}
	if err != nil {
		return err
	}
	if status == behaviortree.Failure {
		return fmt.Errorf("failed to shutdown systemd unit: SubState='%s' ActiveState='%s' UnitFileState='%s' DaemonReloaded='%t' Outdated='%t'", st.SubState, st.ActiveState, st.UnitFileState, st.NeedDaemonReload, st.Outdated)
	}
	return err
}

func (s *SystemdShutdown) Prune(name string, kind systemd.UnitKind) {
	if s.unit.Name == name && s.unit.Kind == kind {
		s.unit.Name = ""
	}
}

func (s *SystemdShutdown) Name() string {
	return s.unit.Name
}

func (s *SystemdShutdown) Plan(plan Plan) Plan {
	if s.unit.Name == "" {
		return plan
	}
	plan = append(plan, map[string]string{"systemd.disable": s.unit.FullName()})
	plan = append(plan, map[string]string{"systemd.stop": s.unit.FullName()})
	return plan
}

func (s *SystemdShutdown) Description() TaskKind {
	return KindSystemdShutdown
}
