package tasks

import (
	"fmt"

	"a.yandex-team.ru/infra/hostctl/internal/behaviortree"
	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/systemd"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks/systemdstates"
)

func NewSystemdTimerRun(name, revID string, running Condition) *SystemdTimerRun {
	unit := systemd.TimerUnit(name)
	return &SystemdTimerRun{
		revID:   revID,
		running: running,
		unit:    unit,
	}
}

func NewRunSystemdTimer(name, revID string, running Condition) *SystemdTimerRun {
	return NewSystemdTimerRun(name, revID, running)
}

type SystemdTimerRun struct {
	revID   string
	running Condition
	unit    *systemd.Unit
}

/*
for SystemService we need
* enable systemd timer
* ActiveState = active
* reload daemon + restart if NeedReloadDaemon
* restart timer if outdated
*/
func (s *SystemdTimerRun) Execute(e *env.Env, changelog *changelog.ChangeLog) error {
	// skipping empty task
	if s.unit.Name == "" {
		return nil
	}
	st := &systemd.UnitStatus{}
	status, err := systemdstates.RunSystemdTimer()(e, changelog, s.unit, s.revID, st).Tick()
	if err != nil {
		s.running.Unknown(err.Error())
		return err
	}
	if st.SubState.Running() {
		s.running.True(fmt.Sprintf("SubState: %s", st.SubState))
	} else {
		s.running.False(fmt.Sprintf("SubState: %s", st.SubState))
	}
	if status == behaviortree.Failure {
		return fmt.Errorf("failed to run systemd unit: ActiveState='%s' UnitFileState='%s'", st.ActiveState, st.UnitFileState)
	}
	return nil
}

func (s *SystemdTimerRun) Prune(name string) {
}

func (s *SystemdTimerRun) Name() string {
	return s.unit.Name
}

func (s *SystemdTimerRun) Plan(plan Plan) Plan {
	if s.unit.Name == "" {
		return plan
	}
	plan = append(plan, map[string]string{"systemd.daemon-reload": s.unit.FullName()})
	plan = append(plan, map[string]string{"systemd.start[timer]": s.unit.FullName()})
	plan = append(plan, map[string]string{"systemd.enable": s.unit.FullName()})
	return plan
}

func (s *SystemdTimerRun) Description() TaskKind {
	return KindSystemdRun
}
