package systemdstates

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/infra/hostctl/internal/behaviortree"
	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/systemd"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/specutil"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks/helper"
)

const RestartThrottleTimeout = 15 * time.Minute

// SystemdServiceUpdater encapsulates update (restart/reload) and active (active/active after reload) methods
// used for SystemService activation.
type SystemdServiceUpdater struct {
	UpdateMethod      func(time.Duration) NewSystemdNode
	ActiveCheckMethod func() StatusNode
}

// ActivateService tries to activate SystemService using supplied updater and retry policy.
// Method uses asynchronous/busywait scheme polling service state after control action to avoid long meaningless sleep
// according to retryPolicy.Timeout
func ActivateService(retryPolicy specutil.RetryPolicy, updater SystemdServiceUpdater) NewSystemdNode {
	return func(e *env.Env, ch *changelog.ChangeLog, u *systemd.Unit, revID string, status *systemd.UnitStatus) behaviortree.Node {
		errMsg := fmt.Sprintf("failed to restart/reload/start systemd service after %d attempts", retryPolicy.Retries)
		return behaviortree.New(
			behaviortree.Fallback,
			ServiceUpToDate(updater.ActiveCheckMethod)(e, ch, u, revID, status),
			helper.WithRetryN(
				behaviortree.New(
					behaviortree.Sequence,
					UpdateServiceFun(updater.UpdateMethod, retryPolicy.Timeout)(e, ch, u, revID, status),
					helper.BusyWaitTimeout(behaviortree.New(behaviortree.Sequence,
						UpdateStatus()(e, ch, u, revID, status),
						ServiceUpToDate(updater.ActiveCheckMethod)(e, ch, u, revID, status),
					), e, retryPolicy.Timeout),
				), errMsg, int(retryPolicy.Retries)),
		)
	}
}

/*
               [?]
                |
        v--------------v
[ServiceUpToDate] [->x3(retry)]
                       |
        v--------------v---------------v
[UpdateService] [UpdateStatus] [ServiceUpToDate]
*/
func ActivateTimer() NewSystemdNode {
	return func(e *env.Env, ch *changelog.ChangeLog, u *systemd.Unit, revID string, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(
			behaviortree.Fallback,
			ServiceUpToDate(ServiceActive)(e, ch, u, revID, status),
			behaviortree.New(
				behaviortree.Sequence,
				UpdateServiceFun(RestartService, 0)(e, ch, u, revID, status),
				UpdateStatus()(e, ch, u, revID, status),
				ServiceUpToDate(ServiceActive)(e, ch, u, revID, status),
			),
		)
	}
}

/*
               [?]
                |
        v----------------------v
[JobStatusOk]                 [->]
                               |
               v-----------v---------------v
              [?] [BusyWaitJobFinish] [JobStatusOk]
               |
        v---------------v
[JobFinishedOk] [UpdateService]
*/
func StartJob(updateMethod func(time.Duration) NewSystemdNode) NewSystemdNode {
	return func(e *env.Env, ch *changelog.ChangeLog, u *systemd.Unit, revID string, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(
			behaviortree.Fallback,
			JobStatusOk()(e, u, status),
			behaviortree.New(
				behaviortree.Sequence,
				// Start/restart/reload only if failed/outdated do not start if not finished yet
				// Not finished status may caused by race condition with timer runs.
				// On collision with timer, BusyWait graceful period before fail.
				behaviortree.New(
					behaviortree.Fallback,
					JobFinishedOk()(e, u, status),
					UpdateServiceFun(updateMethod, 0)(e, ch, u, revID, status),
				),
				BusyWaitJobFinish()(e, ch, u, revID, status),
				JobStatusOk()(e, u, status),
			),
		)
	}
}

func ServiceUpToDate(activeCheckMethod func() StatusNode) NewSystemdNode {
	return func(e *env.Env, ch *changelog.ChangeLog, u *systemd.Unit, revID string, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(
			behaviortree.Sequence,
			activeCheckMethod()(e, u, status),
			ServiceActual()(e, u, status),
		)
	}
}

func JobFinished() StatusNode {
	return func(e *env.Env, u *systemd.Unit, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(func(children []behaviortree.Node) (behaviortree.Status, error) {
			e.L.Infof("Check job finished Exited='%t' (%s) Duration='%s'", !status.JobRunning(), status.ActiveState, status.Duration())
			if status.JobRunning() {
				return behaviortree.Failure, nil
			}
			return behaviortree.Success, nil
		})
	}
}

func JobFinishedOk() StatusNode {
	return func(e *env.Env, u *systemd.Unit, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(
			behaviortree.Sequence,
			JobFinished()(e, u, status),
			JobStatusOk()(e, u, status),
		)
	}
}

func JobStatusOk() StatusNode {
	return func(e *env.Env, u *systemd.Unit, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(func(children []behaviortree.Node) (behaviortree.Status, error) {
			e.L.Infof("Check job status ok (%s) Outdated='%t' ExecMainStatus='%s' Duration='%s'", status.ActiveState, status.Outdated, status.ExecMain.ExecMainStatus, status.Duration())
			if status.Outdated {
				return behaviortree.Failure, nil
			}
			// Restart on failure
			if status.ExecMain.ExecMainStatus != "0" {
				restartAfter := status.ExecMain.Stop.Add(RestartThrottleTimeout)
				// Throttle restarts. We do not want to ddos smth with our 5min restarts.
				if time.Now().Before(restartAfter) {
					return behaviortree.Failure, fmt.Errorf("unit exit code = %s, will restart after %s", status.ExecMain.ExecMainStatus, restartAfter.Format("15:04:05"))
				}
				return behaviortree.Failure, nil
			}
			return behaviortree.Success, nil
		})
	}
}

// BusyWaitJobFinish 30 checks status times every second
func BusyWaitJobFinish() NewSystemdNode {
	return func(e *env.Env, ch *changelog.ChangeLog, u *systemd.Unit, revID string, status *systemd.UnitStatus) behaviortree.Node {
		return helper.BusyWait(
			behaviortree.New(
				behaviortree.Sequence,
				UpdateStatus()(e, ch, u, revID, status),
				JobFinished()(e, u, status),
			), e)
	}
}

func ServiceActual() StatusNode {
	return func(e *env.Env, u *systemd.Unit, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(func(children []behaviortree.Node) (behaviortree.Status, error) {
			e.L.Infof("Check service actual Outdated='%t'", status.Outdated)
			if !status.Outdated {
				return behaviortree.Success, nil
			}
			return behaviortree.Failure, nil
		})
	}
}

func ServiceActive() StatusNode {
	return func(e *env.Env, u *systemd.Unit, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(func(children []behaviortree.Node) (behaviortree.Status, error) {
			e.L.Infof("Check service active ActiveState='%s'", status.ActiveState)
			if status.ActiveState.Active() {
				return behaviortree.Success, nil
			}
			return behaviortree.Failure, nil
		})
	}
}

// ServiceActiveAfterReload checks if service became active after reload.
func ServiceActiveAfterReload() StatusNode {
	return func(e *env.Env, u *systemd.Unit, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(func(children []behaviortree.Node) (behaviortree.Status, error) {
			e.L.Infof("Check service active ActiveState='%s'", status.ActiveState)
			if status.ActiveState.ReloadedActive() {
				return behaviortree.Success, nil
			}
			return behaviortree.Failure, nil
		})
	}
}

// UpdateServiceFun updates or starts service with provided timeout value for systemd interaction (or default timeout)
func UpdateServiceFun(updateFun func(time.Duration) NewSystemdNode, timeout time.Duration) NewSystemdNode {
	timeout = specutil.TimeoutOrDefault(timeout)
	return func(e *env.Env, ch *changelog.ChangeLog, u *systemd.Unit, revID string, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(func(children []behaviortree.Node) (behaviortree.Status, error) {
			if status.ActiveState.Active() {
				return updateFun(timeout)(e, ch, u, revID, status).Tick()
			} else {
				return StartService(timeout)(e, ch, u, revID, status).Tick()
			}
		})
	}
}

// StartService starts systemd service using provided timeout for systemd interaction (or default timeout)
func StartService(timeout time.Duration) NewSystemdNode {
	timeout = specutil.TimeoutOrDefault(timeout)
	return func(e *env.Env, ch *changelog.ChangeLog, u *systemd.Unit, revID string, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(func(children []behaviortree.Node) (behaviortree.Status, error) {
			ch.Add("systemd.start", u.FullName()).Log(e.L)
			ctx, cancel := context.WithTimeout(context.Background(), timeout)
			defer cancel()
			if err := e.Systemd.Start(ctx, u, revID); err != nil {
				e.L.Errorf("Failed to start '%s': %s", u.FullName(), err)
				return behaviortree.Failure, err
			}
			return behaviortree.Success, nil
		})
	}
}

// RestartService restarts systemd service using provided timeout for systemd interaction (or default timeout)
func RestartService(timeout time.Duration) NewSystemdNode {
	timeout = specutil.TimeoutOrDefault(timeout)
	return func(e *env.Env, ch *changelog.ChangeLog, u *systemd.Unit, revID string, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(func(children []behaviortree.Node) (behaviortree.Status, error) {
			ch.Add("systemd.restart", u.FullName()).Log(e.L)
			ctx, cancel := context.WithTimeout(context.Background(), timeout)
			defer cancel()
			if err := e.Systemd.Restart(ctx, u, revID); err != nil {
				e.L.Errorf("Failed to restart '%s': %s", u.FullName(), err)
				return behaviortree.Failure, err
			}
			return behaviortree.Success, nil
		})
	}
}

// ReloadService restarts systemd service using provided timeout for systemd interaction (or default timeout)
func ReloadService(timeout time.Duration) NewSystemdNode {
	timeout = specutil.TimeoutOrDefault(timeout)
	return func(e *env.Env, ch *changelog.ChangeLog, u *systemd.Unit, revID string, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(func(children []behaviortree.Node) (behaviortree.Status, error) {
			ch.Add("systemd.reload", u.FullName()).Log(e.L)
			ctx, cancel := context.WithTimeout(context.Background(), timeout)
			defer cancel()
			if err := e.Systemd.Reload(ctx, u, revID); err != nil {
				e.L.Errorf("Failed to reload '%s': %s", u.FullName(), err)
				return behaviortree.Failure, err
			}
			return behaviortree.Success, nil
		})
	}
}
