package systemdstates

import (
	"context"
	"time"

	"a.yandex-team.ru/infra/hostctl/internal/behaviortree"
	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/systemd"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/tasks/helper"
)

/*
                      [?]
        v----------------------------v
[ServiceDeactivated]           [->x3(retry)]
                                     |
      v------------v---------v----------------v
[StopService] [Sleep] [UpdateStatus] [ServiceDeactivated]
*/
func DeactivateService(timeout time.Duration) NewSystemdNode {
	return func(e *env.Env, ch *changelog.ChangeLog, u *systemd.Unit, revID string, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(
			behaviortree.Fallback,
			// do not need to update status here
			// ActiveState should be updated after ServiceReadyWithUpdate of first state
			ServiceDeactivated()(e, u, status),
			behaviortree.New(
				behaviortree.Sequence,
				StopService(timeout)(e, ch, u, revID, status),
				helper.BusyWaitTimeout(
					behaviortree.New(
						behaviortree.Sequence,
						UpdateStatus()(e, ch, u, revID, status),
						ServiceDeactivated()(e, u, status),
					), e, timeout),
			),
		)
	}
}

func ServiceDeactivated() StatusNode {
	return func(e *env.Env, u *systemd.Unit, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(func(_ []behaviortree.Node) (behaviortree.Status, error) {
			e.L.Infof("Check service inactive ActiveState='%s'", status.ActiveState)
			if !status.ActiveState.Active() {
				return behaviortree.Success, nil
			}
			return behaviortree.Failure, nil
		})
	}
}

func StopService(timeout time.Duration) NewSystemdNode {
	return func(e *env.Env, ch *changelog.ChangeLog, u *systemd.Unit, revID string, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(func(_ []behaviortree.Node) (behaviortree.Status, error) {
			ch.Add("systemd.stop", u.FullName()).Log(e.L)
			ctx, cancel := context.WithTimeout(context.Background(), timeout)
			defer cancel()
			if err := e.Systemd.Stop(ctx, u); err != nil {
				e.L.Errorf("Failed to stop '%s': %s", u.FullName(), err)
				return behaviortree.Failure, err
			}
			return behaviortree.Success, nil
		})
	}
}
