package systemdstates

import (
	"errors"
	"fmt"
	"time"

	"a.yandex-team.ru/infra/hostctl/internal/behaviortree"
	"a.yandex-team.ru/infra/hostctl/internal/changelog"
	"a.yandex-team.ru/infra/hostctl/internal/systemd"
	"a.yandex-team.ru/infra/hostctl/internal/units/env"
	"a.yandex-team.ru/infra/hostctl/internal/units/specutil"
)

var errNotOneshot = errors.New("invalid systemd unit. should be oneshot type")

/*
                      [?]
              v----------------------|
             [->]                    v
      v--------------v              [->]
[UpdateStatus] [ServiceReady]        |
      v------------v-------------v---------------v--------------v
[ReloadDaemon] [ActivateService] [EnableService] [UpdateStatus] [ServiceReady]
*/
func RunSystemdService(retryPolicy specutil.RetryPolicy, updater SystemdServiceUpdater) NewSystemdNode {
	return func(e *env.Env, ch *changelog.ChangeLog, u *systemd.Unit, revID string, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(
			behaviortree.Fallback,
			behaviortree.New(behaviortree.Sequence,
				UpdateStatus()(e, ch, u, revID, status),
				ServiceReady()(e, u, status),
			),
			behaviortree.New(
				behaviortree.Sequence,
				ReloadDaemon()(e, ch, u, revID, status),
				ActivateService(retryPolicy, updater)(e, ch, u, revID, status),
				EnableService()(e, ch, u, revID, status),
				UpdateStatus()(e, ch, u, revID, status),
				ServiceReady()(e, u, status),
			),
		)
	}
}

/*
                      [?]
              v----------------------|
             [->]                    v
      v--------------v              [->]
[UpdateStatus] [ServiceReady]        |
      v------------v-------------v---------------v--------------v
[ReloadDaemon] [ActivateTimer] [EnableService] [UpdateStatus] [ServiceReady]
*/
func RunSystemdTimer() NewSystemdNode {
	return func(e *env.Env, ch *changelog.ChangeLog, u *systemd.Unit, revID string, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(
			behaviortree.Fallback,
			behaviortree.New(behaviortree.Sequence,
				UpdateStatus()(e, ch, u, revID, status),
				ServiceReady()(e, u, status),
			),
			behaviortree.New(
				behaviortree.Sequence,
				ReloadDaemon()(e, ch, u, revID, status),
				ActivateTimer()(e, ch, u, revID, status),
				EnableService()(e, ch, u, revID, status),
				UpdateStatus()(e, ch, u, revID, status),
				ServiceReady()(e, u, status),
			),
		)
	}
}

/*
                      [?]
              v----------------------|
             [->]                    v
      v--------------v              [->]
[UpdateStatus] [JobOk]               |
      v------------v------------v---------------v--------------v
[ReloadDaemon] [StartJob] [EnableService] [UpdateStatus] [JobOk]
*/
func RunOneShotJob(updateMethod func(time.Duration) NewSystemdNode) NewSystemdNode {
	return func(e *env.Env, ch *changelog.ChangeLog, u *systemd.Unit, revID string, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(
			behaviortree.Fallback,
			behaviortree.New(behaviortree.Sequence,
				UpdateStatus()(e, ch, u, revID, status),
				JobOk()(e, u, status),
				behaviortree.New(behaviortree.Fallback,
					JobFinished()(e, u, status),
					BusyWaitJobFinish()(e, ch, u, revID, status),
					behaviortree.New(behaviortree.Sequence,
						FailIfRunning()(e, u, status),
						JobOk()(e, u, status),
					),
				),
			),
			behaviortree.New(
				behaviortree.Sequence,
				ReloadDaemon()(e, ch, u, revID, status),
				StartJob(updateMethod)(e, ch, u, revID, status),
				EnableService()(e, ch, u, revID, status),
				UpdateStatus()(e, ch, u, revID, status),
				JobOk()(e, u, status),
				FailIfRunning()(e, u, status),
			),
		)
	}
}

func ServiceReady() StatusNode {
	return func(e *env.Env, u *systemd.Unit, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(func(_ []behaviortree.Node) (behaviortree.Status, error) {
			e.L.Infof("Check service status ok SubState='%s' ActiveState='%s' UnitFileState='%s' NeedDaemonReload='%t' Outdated='%t'", status.SubState, status.ActiveState, status.UnitFileState, status.NeedDaemonReload, status.Outdated)
			if status.Ok() {
				return behaviortree.Success, nil
			}
			return behaviortree.Failure, nil
		})
	}
}

func JobOk() StatusNode {
	return func(e *env.Env, u *systemd.Unit, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(func(_ []behaviortree.Node) (behaviortree.Status, error) {
			if status.Type != systemd.TypeOneshot {
				return behaviortree.Failure, errNotOneshot
			}
			e.L.Infof("Check job status ok Exited='%t' Outdated='%t' ExecMainStatus='%s' Duration='%s'",
				!status.JobRunning(), status.Outdated, status.ExecMain.ExecMainStatus, status.Duration())
			if !status.JobOk() {
				return behaviortree.Failure, nil
			}
			// will restart if fail
			if status.ExecMain.ExecMainStatus != "0" {
				return behaviortree.Failure, nil
			}
			return behaviortree.Success, nil
		})
	}
}

func FailIfRunning() StatusNode {
	return func(e *env.Env, u *systemd.Unit, status *systemd.UnitStatus) behaviortree.Node {
		return behaviortree.New(func(_ []behaviortree.Node) (behaviortree.Status, error) {
			e.L.Infof("Check job finished fail if still running Exited='%t' Duration='%s'", !status.JobRunning(), status.Duration())
			// We want to avoid "not ready" reporting after job restart,
			// thus we need to wait for some time until it has finished.
			// But it can take longer (and it is okay).
			// Current hypothesis is that most jobs would finish.
			if status.JobRunning() {
				return behaviortree.Failure, fmt.Errorf("job '%s' not finished", u.FullName())
			}
			return behaviortree.Success, nil
		})
	}
}
