// Package hostinfo contains utilities to work with host info, used in commands
package hostinfo

import (
	"errors"
	"fmt"
	"os"
	"strings"

	pb "a.yandex-team.ru/infra/hostctl/proto"
	"github.com/golang/protobuf/jsonpb"
)

// FromEnv tries to load host info as JSON from provided environment variable.
func FromEnv(envName string) (*pb.HostInfo, error) {
	r := strings.NewReader(os.Getenv(envName))
	hi := &pb.HostInfo{}
	err := jsonpb.Unmarshal(r, hi)
	if err != nil {
		return nil, err
	}
	return hi, nil
}

// FromString tries to load host info as JSON from string or `HOST_INCLUDE` environment variable.
func FromString(infoStr string) (*pb.HostInfo, error) {
	var info *pb.HostInfo
	var err error
	if infoStr == "<env>" {
		info, err = FromEnv("HOST_INCLUDE")
		if err != nil {
			return nil, fmt.Errorf("failed to read info from env: %w", err)
		}
	} else if infoStr == "" {
		f, err := os.Open(ServerInfoFile)
		if err != nil {
			return nil, fmt.Errorf("failed to open %s as server_info.json: %w", ServerInfoFile, err)
		}
		defer f.Close()
		info, err = FromStatReader(f)
		if err != nil {
			return nil, fmt.Errorf("can extract host info from /etc/server_info.json: %w", err)
		}
	} else {
		return nil, errors.New("host info from JSON not implemented yet")
	}
	return info, nil
}
